<?php
/**
 * SLM Delivery Service
 * 
 * @author Skroutz Last Mile
 * @license GPL-2.0-or-later
 * 
 * @wordpress-plugin
 * Plugin Name: SLM Delivery Service
 * Description: Integrate SLM locker map into WooCommerce for seamless e-shop navigation and convenience.
 * Text Domain: slm-delivery-service
 * Version: 1.1.13
 * Author: Skroutz Last Mile
 * Author URI: https://www.skroutzlastmile.gr/
 * License: GPL v2 or later
*/

$current_slm_db_version = '1.1';

function slmdb_install() {
	global $wpdb;
	global $current_slm_db_version;

	$table_name = $wpdb->prefix . 'slm_orders';
	
	$charset_collate = $wpdb->get_charset_collate();

	$sql = "CREATE TABLE IF NOT EXISTS $table_name (
		id mediumint(9) NOT NULL AUTO_INCREMENT,
		`order_id` INT NOT NULL,
		`soid` VARCHAR(255),
		`recipient_name` VARCHAR(255) NOT NULL,
		`recipient_phone` VARCHAR(255) NOT NULL,
		`tracking_id` VARCHAR(255),
		`externalLockerId` VARCHAR(255) NOT NULL,
		`status_message` VARCHAR(255) DEFAULT NULL,
		`status` INT DEFAULT 1,
		`slm_tracking_msg` VARCHAR(255) DEFAULT NULL,
		`slm_tracking_status` VARCHAR(255) DEFAULT NULL,
		`pickup_location_code` VARCHAR(255),
		`pickup_date` VARCHAR(255),
		`pickup_time_from` VARCHAR(255),
		`pickup_time_to` VARCHAR(255),
		`pickup_time_enum` ENUM('morning','evening') DEFAULT 'morning',
		created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ,
  		updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
		locker_name VARCHAR(255),
		PRIMARY KEY  (id)
	) $charset_collate;";

	$wpdb->query($sql);

	add_option( 'slm_db_version', $current_slm_db_version );
	add_option( 'slm_last_sync', '' );
	
}

register_activation_hook(__FILE__, 'slmdb_install');


add_action( 'plugins_loaded', 'db_plugin_update' );
function db_plugin_update() {
	global $wpdb;
	global $current_slm_db_version;

	$previous_slm_db_version = (string) get_option( 'slm_db_version' );


    if ( $previous_slm_db_version != $current_slm_db_version ){
		$table_name = $wpdb->prefix . 'slm_orders';

		try {
			$sql = 
			"ALTER TABLE $table_name 
			ADD `locker_name` varchar(255);
			";
	
			$wpdb->query($sql);

			update_option('slm_db_version', $current_slm_db_version);
			
			$slm_last_sync_time = get_option('slm_last_sync');
			if( !isset( $slm_last_sync_time ) ){
				add_option( 'slm_last_sync', '' );
			}

		} catch (Exception $e) {
			// echo 'Caught exception: ',  $e->getMessage(), "\n";
		}
	}
}

function isSubstringInText($string, $substring) {
    if (!isset($string) || !isset($substring) || empty($string) || empty($substring)) {
        return false;
    }

    if (!is_string($string) || !is_string($substring)) {
        return false;
    }

    return strpos($string, $substring) !== false;
}

function slmdb_uninstall() {
	global $wpdb;
	$table_name = $wpdb->prefix . 'slm_orders';
	$sql = "DROP TABLE $table_name";
	$wpdb->query($sql);	

	$options = wp_load_alloptions();
	foreach ($options as $slug => $values) {
		if(isSubstringInText($slug,'slm_')){
			delete_option( $slug );
		}
	}

}


register_uninstall_hook(__FILE__, 'slmdb_uninstall' );


function add_slm_css() {
    $plugin_url = plugin_dir_url( __FILE__ );
    wp_enqueue_style( 'style',  $plugin_url . "/css/slm-styles1113.css");
}

add_action( 'wp_enqueue_scripts', 'add_slm_css' );

function slmdb_create_order($initial_data) {
	global $wpdb;
	$table_name = $wpdb->prefix . 'slm_orders';
	
	$wpdb->insert( 
		$table_name, 
		array( 
			"order_id" => $initial_data["customer_reference"], //client
			"externalLockerId" =>  $initial_data["skroutz_point_id"], //client
			"recipient_name" =>  $initial_data["recipient_name"],  //create voucher
			"recipient_phone" => $initial_data["recipient_phone"],  //create voucher	
			"locker_name" => $initial_data["slm_friendly_name_locker"],  //client
		) 
	);
}


function slm_map(){
	return '
	
			<div id="skroutzLockerMap"></div>
			<div class="slm-wc-fields-hidden">
				<input type="hidden" class="slm-wc-fields-hidden" name="slm_external_id" id="slm_external_id" value="">
			</div>
			<div class="slm-wc-fields-hidden">
				<input type="hidden" class="slm-wc-fields-hidden" name="slm_friendly_name_locker" id="slm_friendly_name_locker" value="">
			</div>
			<script type="text/javascript">
				//slmv: 1.1.13
				var _skroutzPointsMapWidgetOptions = {
					partnerId: "'.get_option('slm_partner_id', '').'",
					type: "iframe",
					parentElement: "#skroutzLockerMap",
					afterSelect: function (selected) {

						if(!selected){
							jQuery("#slm_external_id").val(null);
							jQuery(".slm_selected_locker_msg").html("");
							return;
						} else if(selected.id && selected.name) {
							const data = {locker_id:selected.id,locker_name: selected.name};
							jQuery("#slm_external_id").val(data.locker_id);
							jQuery("#slm_friendly_name_locker").val(data.locker_name);
							jQuery("#locker_id_input").val(data.locker_id);
							
							jQuery(".slm_selected_locker_msg").html("<b>Επιλεγμένο Skroutz Point:</b><br> " + data.locker_name);
						}
						
					}
				};

				(function (d) {
					var e = d.createElement("script");
					e.src = "'.get_option('slm_map_url', '').'";
					e.async = true;
					e.defer = true;
					e.requestUserLocation = true;
					d.getElementsByTagName("head")[0].appendChild(e);
				})(document);

				if(document.querySelector(".closeMapBtn")){
					document.querySelector(".closeMapBtn").addEventListener("click", function () {
						jQuery(".plugin-slm-map .popup-place-slm-map").hide();
						jQuery(".plugin-slm-map .popup-place-slm-map-container").hide();
						// jQuery("body").removeClass("disabled");
	
					});
				}

				var mouse_is_inside = false;
				jQuery(document).ready(function()
				{	
					jQuery("#skroutzLockerMap").hover(function(){ 
						mouse_is_inside=true;
					}, function(){ 
						mouse_is_inside=false; 
					});

					jQuery("body").mouseup(function(){ 
						if(! mouse_is_inside) {
							jQuery(".plugin-slm-map .popup-place-slm-map").hide();
							jQuery(".plugin-slm-map .popup-place-slm-map-container").hide();
							// jQuery("body").removeClass("disabled");
						}
					});

					jQuery("#skroutzLockerMap iframe").each(function(index){
						if (index !== 0) {
							jQuery(this).remove(); // Handles a undefined bug that duplicates the map
						}
					});
				});
			</script>

			<input type="hidden" name="locker_id_input">
				
			
			';
}

// add_menu_page->( string $page_title, string $menu_title, string $capability, string $menu_slug, callable $callback = '', string $icon_url = '', int|float $position = null ): string
function link_slm_orders_tab(){
    add_menu_page( 'SLM Orders', 'SLM Orders', 'read', 'slm-orders', 'slm_orders_page', 'dashicons-location');
}

function slm_orders_page(){
	include_once('includes/slm-orders/wc-slm-orders-page.php');
}

add_action('admin_menu', 'link_slm_orders_tab');

function woo_shipping_settings_tab($tabs) {

	$tabs['slm_service_settings'] = __('SLM Settings', 'woo-slm-delivery-service');

	return $tabs;
}

function getWCOrderStatuses(){
	return wc_get_order_statuses();
}

function getWCcategories(){

	$categories = get_terms( ['taxonomy' => 'product_cat'] );
	return $categories;
}

function woo_shipping_hooks() {
	// Add WC settings tab
	add_filter('woocommerce_settings_tabs_array', 'woo_shipping_settings_tab', 70);

	// Settings page contents
	include_once('includes/slm-settings/wc-slm-settings-page.php');
	add_action('woocommerce_settings_tabs_slm_service_settings', 'woo_shipping_settings_page');

	// Save settings page
	include_once('includes/slm-settings/wc-slm-update-options.php');
	add_action('woocommerce_update_options_slm_service_settings', 'woo_shipping_update_options');
}

woo_shipping_hooks();

if(in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))){
	function slm_shipping_method() {
		if(!class_exists('Slm_shipping_method')){
			class Slm_shipping_method extends WC_Shipping_Method {

                /**
                 * @var array Shipping method form fields
                 */
                public $instance_form_fields = array();

                public function __construct($instance_id = 0) {
					$this->id                 = 'slm_delivery_service';
					$this->instance_id        = absint( $instance_id );
					$this->method_title       = __( 'SLM', 'woocommerce' );
					$this->method_description = __( 'Offer clients reliable, convenient SLM Lockers for seamless and efficient shipping experiences.', 'woocommerce' );
					$this->supports           = array(
						'shipping-zones',
						'instance-settings',
						'instance-settings-modal',
					);
					$this->init();
				}

				function init() {

					$this->set_wc_options_for_shipping_method_slm(); 
					$this->init_settings(); 

					$this->title       = $this->get_option( 'title' );
					$this->tax_status  = $this->get_option( 'tax_status' );
					$this->cost        = $this->get_option( 'cost' );
					$this->freethreshold    = $this->get_option( 'freethreshold' );


					add_action( 'woocommerce_update_options_shipping_' . $this->id, array( $this, 'process_admin_options' ) );
				}

				/**
                 * Calculate shipping rates
                 *
                 * @param array $package Shipping package details.
                 */
				public function calculate_shipping( $package = array() ) {
						$subtotal = WC()->cart->subtotal;
                    	$cost     = (isset($this->freethreshold) && !empty($this->freethreshold) && $subtotal >= $this->freethreshold) ? 0 : $this->cost;

						$this->add_rate(
							array(
								'label'   => $this->title,
								'package' => $package,
								'cost'    => $cost,
							)
						);
						
				}

				/**
                 * Set WooCommerce options for the shipping method
                 */
				public function set_wc_options_for_shipping_method_slm() {
                    $this->instance_form_fields = array(
                        'title'         => array(
                            'title'       => esc_html__('Title', 'woocommerce'),
                            'type'        => 'text',
                            'description' => esc_html__('This dictates what the title will be called for the user when they proceed to check out.', 'woocommerce'),
                            'default'     => esc_html__('SLM Locker Delivery', 'woocommerce'),
                            'desc_tip'    => true,
                        ),
                        'tax_status'    => array(
                            'title'       => esc_html__('Tax status', 'woocommerce'),
                            'type'        => 'select',
                            'class'       => 'wc-enhanced-select',
                            'description' => esc_html__('Enable Shipping Cost to be taxable, with the applied Tax Class if any.', 'woocommerce'),
                            'default'     => 'none',
                            'options'     => array(
                                'taxable' => esc_html__('Taxable', 'woocommerce'),
                                'none'    => esc_html__('None', 'woocommerce'),
                            ),
                            'desc_tip'    => true,
                        ),
                        'cost'          => array(
                            'title'       => esc_html__('Cost', 'woocommerce'),
                            'type'        => 'text',
                            'placeholder' => '0',
                            'description' => esc_html__('Base fee for SLM delivery, please change this.', 'woocommerce'),
                            'default'     => '',
                            'desc_tip'    => true,
                        ),
                        'freethreshold' => array(
                            'title'       => esc_html__('Free delivery', 'woocommerce'),
                            'type'        => 'text',
                            'placeholder' => '',
                            'description' => esc_html__('Shipping is free for orders exceeding this amount. Leave blank for no free shipping.', 'woocommerce'),
                            'default'     => '',
                            'desc_tip'    => true,
                        ),
                    );
                }
			}
		}
	}

	function add_slm_shipping_method( $methods ) {
		$methods['slm_delivery_service'] = 'Slm_shipping_method';
		return $methods;
	}

	add_filter( 'woocommerce_shipping_methods', 'add_slm_shipping_method' );

	add_action( 'woocommerce_shipping_init', 'slm_shipping_method' );

	

	function cart_not_contains_included_category(){
		$cart = WC()->cart;

		$slm_include_categories = explode(',',get_option('slm_include_categories'));
		$slm_soft_include = get_option('slm_soft_include');

		if(!isset($slm_soft_include) || empty($slm_soft_include)){
			$slm_soft_include = false;
		} else{
			$slm_soft_include = true;
		}

		// Loop through cart items
		foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
			$dont_show_slm = $slm_soft_include; //if slm is hard include then this will be false

			$product_id = $cart_item['product_id'];
			$category_of_item = get_the_terms($product_id, 'product_cat');

			if ($category_of_item && !is_wp_error($category_of_item)) {
				
				foreach ($category_of_item as $category) {

					if($slm_soft_include){
						if(in_array($category->term_id, $slm_include_categories) ){
							$dont_show_slm = false;
							break;
						}
					} else {
						if(!in_array($category->term_id, $slm_include_categories) ){
							return true;
						}
					}

				}

				if($dont_show_slm === true){
					return true;
				}
			}
		}

		return false;
	}


	function disable_shipping_methods( $rates, $package ) {

		
		$show_disabled_slm_msg = ( is_cart() || is_checkout() ) && isSubstringInText(WC()->session->get( 'chosen_shipping_methods' )[0], 'slm_delivery_service');

		$disable_slm = false;

		$total_weight = calculateCartTotalWeight();
		if($total_weight > 25){

			if($show_disabled_slm_msg){
				wc_add_notice('Οι παραγγελίες με συνολικό βάρος άνω των 25 κιλών δεν μπορούν να παραδοθούν σε Skroutz Point.', 'error');
			}

			
			$shipping_method_to_remove = 'slm_delivery_service';
			foreach( $rates as $rate_id => $rate ) {
				if ( $rate->method_id === $shipping_method_to_remove ) {
					unset( $rates[$rate_id] );
				} 
			}
			$disable_slm = true;
		} else if(cart_not_contains_included_category()){
			if($show_disabled_slm_msg){
				wc_add_notice('Ορισμένα προϊόντα στο καλάθι δεν μπορούν να παραδοθούν σε Skroutz Point', 'error');
			}
			
			$disable_slm = true;
		}

		if($disable_slm){
			$shipping_method_to_remove = 'slm_delivery_service';
			foreach( $rates as $rate_id => $rate ) {
				if ( $rate->method_id === $shipping_method_to_remove ) {
					unset( $rates[$rate_id] );
				} 
			}
		}

		return $rates;

	}
	
	add_filter( 'woocommerce_package_rates', 'disable_shipping_methods', 100, 2 );

	function custom_disable_payment_method_based_on_shipping($gateways) {
        // Check if the WC()->session is set and contains the chosen shipping methods
        if (isset(WC()->session) && WC()->session->get('chosen_shipping_methods')) {
            $chosen_shipping_methods = WC()->session->get('chosen_shipping_methods');
    
            // Ensure that the chosen shipping methods is an array and contains at least one method
            if (is_array($chosen_shipping_methods) && !empty($chosen_shipping_methods)) {
                // Use the first chosen shipping method
                $chosen_shipping_method = $chosen_shipping_methods[0];
    
                // Check if the chosen shipping method contains the target substring
                if (strpos($chosen_shipping_method, 'slm_delivery_service') !== false) {
                    unset($gateways['cod']); // Unset the COD payment method
                }
            }
        }
    
        return $gateways;
    }

	if (!is_admin())
	{ 
		add_filter('woocommerce_available_payment_gateways', "custom_disable_payment_method_based_on_shipping"); 
	}

	function add_slm_popup_map($method) {

		$slm_map_type =  empty(get_option('slm_map_type')) ? 'popup' : get_option('slm_map_type');

		if(is_cart() || $slm_map_type != 'popup' || $method->method_id !== 'slm_delivery_service'){
			return;
		} else {
			if(isSubstringInText(WC()->session->get( 'chosen_shipping_methods' )[0],'slm_delivery_service')) {
				echo '<script>jQuery(".open-map-btn").show();</script>';
			} else {
				echo '<script>
						jQuery(".open-map-btn").hide();
						jQuery("#slm_external_id").val("");
					  </script>';
			}
			echo '<div class="plugin-slm-map">
				<p class="slm_selected_locker_msg"></p>
				<a class="open-map-btn" style="margin-bottom: 10px">Διαλέξτε Skroutz Point</a>
				
				<div class="popup-place-slm-map-container" style="display:none">
					<div class="popup-place-slm-map" style="text-align:right; display:none">
						<span class="closeMapBtn">ⓧ</span>
						'
						.slm_map().
					'</div>
				</div>
			</div>';
			echo '<script>jQuery( document ).ready(function(){
					jQuery(".plugin-slm-map a.open-map-btn").click(function() {
						jQuery(".plugin-slm-map .popup-place-slm-map, .plugin-slm-map .popup-place-slm-map #slm_integration").toggle();
						jQuery(".plugin-slm-map .popup-place-slm-map-container").toggle();
						// jQuery("body").toggleClass("disabled");
					});
				});</script>';
			
		}
	}

	add_action( 'woocommerce_after_shipping_rate', 'add_slm_popup_map');

	function add_slm_iframe_map(){

		$slm_map_type =  empty(get_option('slm_map_type')) ? 'popup' : get_option('slm_map_type');
		if($slm_map_type != 'iframe'){
			return null;
		}else{
			
			return '
			<script>
				jQuery(document.body).on( "change", "input.shipping_method", function(e) {
					if(!e.currentTarget.value.includes("slm_delivery_service")){
						jQuery("#skroutzLockerMap").hide("fast");
					} else{
						jQuery("#skroutzLockerMap").show("fast");
					}
				});
			</script>
			
			'
			.

			slm_map()
			.

			'<p class="slm_selected_locker_msg"></p>';
			
			
		}
	}	

	
	add_shortcode('add-slm-map-iframe', 'add_slm_iframe_map');


	function write_slm_fields($order_id) {
		if (isset($_POST['slm_external_id'])){
			update_post_meta($order_id, 'slm_external_id', $_POST['slm_external_id']);
		}
		if(isset($_POST['slm_friendly_name_locker'])){
			update_post_meta($order_id, 'slm_friendly_name_locker', $_POST['slm_friendly_name_locker']);
		}
	}
	add_action('woocommerce_checkout_update_order_meta', 'write_slm_fields');

	function calculateCartTotalWeight() {

		$slm_enable_weight_get_value = get_option('slm_enable_weight');

		if($slm_enable_weight_get_value != 'on') {
			return 0;
		};

		$cart = WC()->cart; 
		$total_weight = 0; 
		$weight_unit = get_option('woocommerce_weight_unit');

		foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
			$product = $cart_item['data'];
			$quantity = (int) $cart_item['quantity']; 
			$weight = (float) $product->get_weight(); 
			$total_weight += $quantity * $weight; 

			//convert to kg if other unit
			if($weight_unit === 'g') {
				$total_weight = $total_weight / 1000;
			} else if($weight_unit === 'lbs'){
				$total_weight = $total_weight * 0.453592;
			} else if($weight_unit === 'oz'){
				$total_weight = $total_weight * 0.0283495;
			}
		}

		return $total_weight;
	}


	function pre_check_slm_order_complete() {

		if(!isSubstringInText(WC()->session->get( 'chosen_shipping_methods' )[0], 'slm_delivery_service')){
			return;
		}

		if(cart_not_contains_included_category()){
			wc_add_notice('Ορισμένα προϊόντα στο καλάθι δεν μπορούν να παραδοθούν σε Skroutz Point', 'error');
		}

		$total_weight = calculateCartTotalWeight();
		if($total_weight > 25){
			wc_add_notice('Οι παραγγελίες με συνολικό βάρος άνω των 25 κιλών δεν μπορούν να παραδοθούν σε Skroutz Point.', 'error');
		}
		// Locker Selected Check
		if ((!$_POST['slm_external_id'])){
			wc_add_notice('Παρακαλώ επιλέξτε Skroutz Point για να συνεχίσετε.','error');
		}


		//Mobile Phone Check
		if (($_POST['billing_phone'])){
			//add logic  for phone to be mobile phone

			$phone = $_POST['billing_phone'];
			$phone = str_replace(' ', '', $phone);
			$phone = str_replace('+', '', $phone);
			
			$re = '/^(?:\+?30|0)?69\d{8}$/m';
			
			if(!preg_match($re, $phone) ){
				wc_add_notice('Παρακαλώ εισάγετε έγκυρο κινητό τηλέφωνο (69xxxxxxxx).','error');
			}
		} else{
			wc_add_notice('Παρακαλώ εισάγετε κινητό τηλέφωνο.','error');
		}
		
	}
	
	add_action('woocommerce_checkout_process', 'pre_check_slm_order_complete');
	add_action( 'woocommerce_new_order', 'wc_slm_order_from_client', 10, 1 );

	function wc_slm_order_from_client( $order )  {
		
		function checkShippingMethod($shipping_method) {

			if (!is_array($shipping_method) || empty($shipping_method)) {
				return false; 
			} 

			foreach ($shipping_method as $method) {
				if (isSubstringInText($method, 'slm_delivery_service')) {
					return true; 
				}

				
			}
			return false; 
		}
		
		if(checkShippingMethod($_POST['shipping_method'])){

			$phone = $_POST['billing_phone'];
			$phone = str_replace(' ', '', $phone);
			$phone = str_replace('+', '', $phone);
			//save data to database
			$data = array(
				"customer_reference"		=> $order,//"order_id from table wp_woocommerce_order_items",
				"recipient_name"			=> $_POST['billing_first_name'] . " " . $_POST['billing_last_name'],
				"recipient_phone"			=> $phone,
				"skroutz_point_id"			=> $_POST['slm_external_id'],
				"slm_friendly_name_locker"	=> $_POST['slm_friendly_name_locker'],
			);
			slmdb_create_order($data);
		}
	}
}
?>