<?php
/*
Plugin Name: Woo Discount to Custom Field
Description: Υπολογίζει το ποσοστό έκπτωσης για κάθε προϊόν και το αποθηκεύει σε custom πεδίο "discount" (συμβατό με JetEngine).
Version: 1.2
Author: ProMarketing.gr
*/

if (!defined('ABSPATH')) exit;

// Υπολογισμός και αποθήκευση ποσοστού έκπτωσης στο custom field 'discount'
function save_discount_to_jetengine_field($post_id) {
    $product = wc_get_product($post_id);
    if (!$product) return;

    $max_discount = 0;

    if ($product->is_type('simple')) {
        $regular_price = (float) $product->get_regular_price();
        $sale_price = (float) $product->get_sale_price();

        if ($regular_price && $sale_price && $regular_price > $sale_price) {
            $max_discount = round((($regular_price - $sale_price) / $regular_price) * 100);
        }

    } elseif ($product->is_type('variable')) {
        $variations = $product->get_children();

        foreach ($variations as $variation_id) {
            $variation = wc_get_product($variation_id);
            if (!$variation) continue;

            $regular_price = (float) $variation->get_regular_price();
            $sale_price = (float) $variation->get_sale_price();

            if ($regular_price && $sale_price && $regular_price > $sale_price) {
                $discount = round((($regular_price - $sale_price) / $regular_price) * 100);
                if ($discount > $max_discount) {
                    $max_discount = $discount;
                }
            }
        }
    }

    update_post_meta($post_id, 'discount', $max_discount);
}

// Ενημέρωση όταν αποθηκεύεται οποιοδήποτε προϊόν (simple ή variable)
add_action('woocommerce_update_product', 'save_discount_to_jetengine_field', 20, 1);

// Ενημέρωση όταν αποθηκεύεται κάποια παραλλαγή
add_action('woocommerce_update_product_variation', 'update_parent_discount_on_variation_update', 20, 1);
function update_parent_discount_on_variation_update($variation_id) {
    $variation = wc_get_product($variation_id);
    if (!$variation) return;

    $parent_id = $variation->get_parent_id();
    if ($parent_id) {
        save_discount_to_jetengine_field($parent_id);
    }
}

// Προσθήκη στο admin menu κάτω από WooCommerce
add_action('admin_menu', 'woo_discount_custom_menu');
function woo_discount_custom_menu() {
    add_submenu_page(
        'woocommerce',
        'Ενημέρωση Εκπτώσεων',
        'Ενημέρωση Εκπτώσεων',
        'manage_woocommerce',
        'woo-discount-update',
        'woo_discount_admin_page'
    );
}

// Περιεχόμενο admin page με κουμπί ενημέρωσης και οδηγίες cron
function woo_discount_admin_page() {
    if (isset($_POST['run_bulk_update']) && current_user_can('manage_woocommerce')) {
        woo_discount_run_bulk_update();
        echo '<div class="updated"><p>Η μαζική ενημέρωση ολοκληρώθηκε!</p></div>';
    }

    echo '<div class="wrap">';
    echo '<h1>Μαζική Ενημέρωση Εκπτώσεων</h1>';
    echo '<form method="post">';
    echo '<p>Πάτησε το παρακάτω κουμπί για να ενημερώσεις το ποσοστό έκπτωσης όλων των προϊόντων.</p>';
    echo '<input type="submit" name="run_bulk_update" class="button-primary" value="Ενημέρωση Όλων των Εκπτώσεων">';
    echo '</form>';

    echo '<hr><h2>Αυτοματοποιημένη Εκτέλεση (Cron Job)</h2>';
	echo '<p>Για να λειτουργήσει το συγκεκριμένο plugin πρέπει να δημιουργηθεί πρώτα, ένα custom field μέσω του JetEngine με την ονομασία "discount"</p>';
	echo '<p>Το συγκεκριμένο plugin συγκρίνει την regular price με την sale price και συμπληρώνει στο custom field discount το ποσοστό έκπτωσης</p>';
    echo '<p>Αν θέλεις να γίνεται η ενημέρωση αυτόματα, μπορείς να καλέσεις την ακόλουθη διεύθυνση από cron job ή εξωτερικό scheduler:</p>';
	echo '<p>wget -q -O - "https://example.com/?woo_discount_cron_update=1" >/dev/null 2>&1</p>';
    echo '<code>' . site_url('/?woo_discount_cron_update=1') . '</code>';
    echo '<p>Μπορείς να την προγραμματίσεις να τρέχει π.χ. κάθε βράδυ.</p>';
    echo '</div>';
}

// Συνάρτηση μαζικής ενημέρωσης όλων των προϊόντων
function woo_discount_run_bulk_update() {
    $args = array(
        'post_type' => ['product', 'product_variation'],
        'post_status' => 'publish',
        'posts_per_page' => -1,
        'fields' => 'ids',
    );
    $products = get_posts($args);

    foreach ($products as $product_id) {
        $product = wc_get_product($product_id);
        if ($product && ($product->is_type('simple') || $product->is_type('variable'))) {
            save_discount_to_jetengine_field($product_id);
        }
    }
}

// Endpoint για εξωτερικό cron job
add_action('init', function() {
    if (isset($_GET['woo_discount_cron_update']) && $_GET['woo_discount_cron_update'] == 1) {
        if (!current_user_can('manage_woocommerce')) return;
        woo_discount_run_bulk_update();
        echo "Μαζική ενημέρωση εκτελέστηκε επιτυχώς.";
        exit;
    }
});
?>
