<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://www.webexpert.gr/
 * @since      1.0.0
 *
 * @package    Webexpert_Woocommerce_Skroutz_Smart_Cart
 * @subpackage Webexpert_Woocommerce_Skroutz_Smart_Cart/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Webexpert_Woocommerce_Skroutz_Smart_Cart
 * @subpackage Webexpert_Woocommerce_Skroutz_Smart_Cart/public
 * @author     Web Expert <info@webexpert.gr>
 */
class Webexpert_Woocommerce_Skroutz_Smart_Cart_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function fetch($path,$method='GET',$query=[]) {
		if (empty(get_option('webexpert_skroutz_smart_cart_api_token',null))) {
			return false;
		}

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch,CURLOPT_RETURNTRANSFER,1);
		curl_setopt($ch,CURLOPT_HTTPHEADER, array(
			"Authorization: Bearer ".get_option('webexpert_skroutz_smart_cart_api_token',null),
			"Content-Type: application/json",
			"Accept: application/vnd.skroutz+json; version=3.0"
		));

		if ($method=="POST") {
			curl_setopt($ch, CURLOPT_POST, 1);
			curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($query,JSON_UNESCAPED_UNICODE));
			$url="https://api.skroutz.gr/merchants/ecommerce/{$path}";
		}elseif ($method=="PUT") {
			curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT");
			curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
			curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($query,JSON_UNESCAPED_UNICODE));
			$url="https://api.skroutz.gr/merchants/ecommerce/{$path}";
		}else {
			$url="https://api.skroutz.gr/merchants/ecommerce/{$path}";
			if (sizeof($query)>0) {
				$url="https://api.skroutz.gr/merchants/ecommerce/{$path}?".http_build_query($query);
			}
		}

		curl_setopt($ch, CURLOPT_URL,$url);

		return json_decode(curl_exec($ch));
	}

    public function register_smart_cart_fn()
    {
        register_rest_route( 'wc/v3', 'smart_cart', array(
            'methods' => 'POST',
            'callback' => array($this,'process_order'),
            'permission_callback' => '__return_true'
        ));
	}

	public function prefecture_to_state($prefecture) {
		$prefecture=wc_strtoupper($prefecture);
		switch ($prefecture) {
			case wc_strtoupper('Αττικής'):
				return 'I';
				break;
			case wc_strtoupper('Αγίου Όρους'):
			case wc_strtoupper('Ημαθίας'):
			case wc_strtoupper('Θεσσαλονίκης'):
			case wc_strtoupper('Κιλκίς'):
			case wc_strtoupper('Πέλλας'):
			case wc_strtoupper('Πιερίας'):
			case wc_strtoupper('Σερρών'):
			case wc_strtoupper('Χαλκιδικής'):
				return 'B';
				break;
			case wc_strtoupper('Αιτωλοακαρνανίας'):
			case wc_strtoupper('Αχαΐας'):
			case wc_strtoupper('Ηλείας'):
				return 'G';
				break;
			case wc_strtoupper('Αργολίδος'):
			case wc_strtoupper('Αρκαδίας'):
			case wc_strtoupper('Κορινθίας'):
			case wc_strtoupper('Λακωνίας'):
			case wc_strtoupper('Μεσσηνίας'):
				return 'J';
				break;
			case wc_strtoupper('Άρτης'):
			case wc_strtoupper('Θεσπρωτίας'):
			case wc_strtoupper('Ιωαννίνων'):
			case wc_strtoupper('Πρεβέζης'):
				return 'D';
				break;
			case wc_strtoupper('Βοιωτίας'):
			case wc_strtoupper('Ευβοίας'):
			case wc_strtoupper('Ευρυτανίας'):
			case wc_strtoupper('Φθιώτιδας'):
			case wc_strtoupper('Φωκίδας'):
				return 'H';
				break;
			case wc_strtoupper('Γρεβενών'):
			case wc_strtoupper('Καστοριάς'):
			case wc_strtoupper('Κοζάνης'):
			case wc_strtoupper('Φλωρίνης'):
				return 'C';
				break;
			case wc_strtoupper('Δράμας'):
			case wc_strtoupper('Έβρου'):
			case wc_strtoupper('Καβάλας'):
			case wc_strtoupper('Ξάνθης'):
			case wc_strtoupper('Ροδόπης'):
				return 'A';
				break;
			case wc_strtoupper('Δωδεκανήσου'):
			case wc_strtoupper('Κυκλάδων'):
				return 'L';
				break;
			case wc_strtoupper('Ζακύνθου'):
			case wc_strtoupper('Κερκύρας'):
			case wc_strtoupper('Κεφαλληνίας'):
			case wc_strtoupper('Λευκάδος'):
				return 'F';
				break;
			case wc_strtoupper('Ηρακλείου'):
			case wc_strtoupper('Ρεθύμνου'):
			case wc_strtoupper('Χανίων'):
			case wc_strtoupper('Λασιθίου'):
				return 'M';
				break;
			case wc_strtoupper('Καρδίτσας'):
			case wc_strtoupper('Λαρίσης'):
			case wc_strtoupper('Μαγνησίας'):
			case wc_strtoupper('Τρικάλων'):
				return 'E';
				break;
			case wc_strtoupper('Λέσβου'):
			case wc_strtoupper('Σάμου'):
			case wc_strtoupper('Χίου'):
				return 'K';
				break;
			default:
				return 'I';
		}
	}

    function add_my_gateway_class( $methods ) {
        $methods[] = 'WC_Smart_Cart_Gateway';
        return $methods;
    }

	function webexpert_skroutz_smart_cart_do_not_reduce_fbs_stock($reduce_stock, $order) {
		if ((strpos($order->get_order_key(), 'SC-') !== false) && !empty($order->get_meta( '_skroutz_fulfilled' ))) {
			$reduce_stock = false;
		}
		return $reduce_stock;
	}

	public function disable_skroutz_emails($recipient, $order)
	{
		if ($order instanceof WC_Order) {
			if ($order->get_created_via() == "skroutz_smart_cart") {
				$recipient = '';
			}
		}
		return $recipient;
	}

	function create_new_order($sc_order, $update=false, $lock_bypass = false) {
		$upload_dir = wp_upload_dir();
		wp_mkdir_p(rtrim($upload_dir['basedir'],'/')."/webexpert-smart-cart");
		$lock_file = rtrim($upload_dir['basedir'],'/')."/webexpert-smart-cart/{$sc_order->code}.lock";

		if (file_exists($lock_file) && !$lock_bypass) {
			return null;
		}

		global $wp_filesystem;
		require_once(ABSPATH . 'wp-admin/includes/file.php');
		WP_Filesystem();

		$wp_filesystem->put_contents($lock_file, 'locked');

        try {
            $preferred_payment_gateway=get_option('webexpert_skroutz_smart_cart_prefered_payment_gateway','skroutz_smart_cart');
            $preferred_shipping_method=get_option('webexpert_skroutz_smart_cart_prefered_shipping_method','');
            $order_id=wc_get_order_id_by_order_key("SC-{$sc_order->code}");
	        if ($order_id) {
		        $order=wc_get_order($order_id);
		        if ( ! $update ) {
			        return null;
		        }
	        }else {
		        $order=new WC_Order();
	        }

            $order->set_order_key("SC-{$sc_order->code}");

	        if (in_array($sc_order->state,["cancelled","rejected","expired"])) {
		        $order->set_status("cancelled");
		        $order->set_status(apply_filters("webexpert_skroutz_smart_cart_custom_cancelled_status","cancelled", $order));
	        }elseif (in_array($sc_order->state,["open"])) {
		        $order->set_status(apply_filters("webexpert_skroutz_smart_cart_custom_on-hold_status","on-hold", $order));
	        }elseif (in_array($sc_order->state,["dispatched","delivered"])) {
		        $order->set_status(apply_filters("webexpert_skroutz_smart_cart_custom_completed_status","completed", $order));
	        }elseif (in_array($sc_order->state,["returned","partially_returned"])) {
		        $order->set_status(apply_filters("webexpert_skroutz_smart_cart_custom_refunded_status","refunded", $order));
	        }elseif (in_array($sc_order->state,["for_return"])) {
		        $order->set_status(apply_filters("webexpert_skroutz_smart_cart_custom_for-return_status","for-return", $order));
	        }else {
		        $order->set_status(apply_filters("webexpert_skroutz_smart_cart_custom_processing_status","processing", $order));
	        }

            $email=get_option('webexpert_skroutz_smart_cart_billing_email','');
			if (empty($email)) {
				$email="{$sc_order->customer->id}@skroutz.gr";
			}
            if (get_option('webexpert_skroutz_smart_cart_user_id',-1)>0) {
                $order->set_customer_id(get_option('webexpert_skroutz_smart_cart_user_id',-1));
                $user_data=get_userdata(get_option('webexpert_skroutz_smart_cart_user_id',-1));
                $email=empty(get_option('webexpert_skroutz_smart_cart_billing_email')) ? $user_data->user_email : get_option('webexpert_skroutz_smart_cart_billing_email');
            }
            $order->set_date_created(strtotime($sc_order->created_at));
            $order->set_date_paid(strtotime($sc_order->created_at));
            $order->set_date_modified(strtotime($sc_order->created_at));
            $order->set_created_via( 'skroutz_smart_cart' );
            $order->set_currency( get_woocommerce_currency() );
            $order->set_prices_include_tax( 'yes' === get_option( 'woocommerce_prices_include_tax' ) );
            $order->set_customer_note( $sc_order->comments ?? '');

            foreach ( WC()->payment_gateways->payment_gateways() as $method ) {
            	if ($method && isset($method->id) && $method->id==$preferred_payment_gateway) {
		            $order->set_payment_method_title( $method->get_method_title() );
	            }
            }
	        $order->set_payment_method( $preferred_payment_gateway );
	        $order->update_meta_data(apply_filters('webexpert_skroutz_smart_cart_customer_id','skroutz_customer_id'),$sc_order->customer->id);

            $order->set_billing_first_name($sc_order->customer->first_name);
            $order->set_billing_last_name($sc_order->customer->last_name);
            $order->set_billing_address_1($sc_order->customer->address->street_name. ' '. $sc_order->customer->address->street_number);
            $order->set_billing_city($sc_order->customer->address->city);
	        $order->set_billing_postcode($sc_order->customer->address->zip);
	        if ( in_array( 'webexpert-greek-states-and-cities/webexpert-greek-states-and-cities.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) || in_array( 'webexpert-greek-states-based-shipping/webexpert-greek-states-and-cities.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
		        $order->set_billing_state($sc_order->customer->address->country_code == 'GR' ? wc_strtoupper($sc_order->customer->address->region) : '');
	        }else {
		        $order->set_billing_state($sc_order->customer->address->country_code == 'GR' ? wc_strtoupper($sc_order->customer->address->region) : '');
	        }
			if (!empty($sc_order->customer->phone)) {
				$order->set_billing_phone($sc_order->customer->phone);
			}
	        if (!empty($sc_order->customer->mobile)) {
		        $order->update_meta_data(apply_filters('webexpert_smart_cart_custom_cellphone','_billing_cellphone'),$sc_order->customer->mobile);
	        }
            $order->update_meta_data('_skroutz_customer_id',$sc_order->customer->id);
            $order->update_meta_data('_skroutz_pickup_from_collection_point',$sc_order->customer->address->pickup_from_collection_point ?? '');
            $order->update_meta_data('_skroutz_collection_point_address',$sc_order->customer->address->collection_point_address ?? '');
            $order->update_meta_data('_skroutz_courier',$sc_order->courier ?? '');
            $order->update_meta_data('_skroutz_express',$sc_order->express ?? '');
            $order->update_meta_data('_skroutz_gift_wrap',$sc_order->gift_wrap ?? '');
            $order->update_meta_data('_skroutz_store_pickup',$sc_order->store_pickup ?? '');
            $order->update_meta_data('_skroutz_fulfilled',$sc_order->fulfilled_by_skroutz ?? '');
            $order->update_meta_data('_skroutz_dispatch_until',$sc_order->dispatch_until ?? '');
            $order->update_meta_data('_skroutz_courier_voucher',$sc_order->courier_voucher ?? '');
            $order->update_meta_data('_skroutz_courier_tracking_codes',$sc_order->courier_tracking_codes ?? '');
	        $order->update_meta_data('_skroutz_commission',$sc_order->commission ?? '');
            $order->set_billing_email(str_replace("{customer}",$sc_order->customer->id,$email));
            $order->set_billing_country($sc_order->customer->address->country_code ?? '');

            $order->set_shipping_first_name($sc_order->customer->first_name ?? '');
            $order->set_shipping_last_name($sc_order->customer->last_name ?? '');
            $order->set_shipping_address_1($sc_order->customer->address->street_name. ' '. $sc_order->customer->address->street_number);
	        $order->set_shipping_city($sc_order->customer->address->city ?? '');
            $order->set_shipping_postcode($sc_order->customer->address->zip ?? '');
	        if ( in_array( 'webexpert-greek-states-and-cities/webexpert-greek-states-and-cities.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) || in_array( 'webexpert-greek-states-based-shipping/webexpert-greek-states-and-cities.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
		        $order->set_shipping_state($sc_order->customer->address->country_code == 'GR' ? wc_strtoupper($sc_order->customer->address->region) : '');
	        }else {
		        $order->set_shipping_state($sc_order->customer->address->country_code == 'GR' ? wc_strtoupper($sc_order->customer->address->region) : '');
	        }
            $order->set_shipping_country($sc_order->customer->address->country_code ?? '');

	        if ( isset( $sc_order->invoice_details ) && ! empty( $sc_order->invoice_details->vat_exclusion_requested ) ) {
		        $order->update_meta_data('is_vat_exempt', 'yes');
	        }

			$order_total=0.0;
	        foreach( $sc_order->line_items as $line_item ) {
		        $order_total+=$line_item->total_price;
	        }
			$order->set_total($order_total);

	        foreach( $sc_order->line_items as $line_item ) {
		        $product=null;
		        $special_id=get_option('we_skroutz_xml_color_id_parent_id',get_option('webexpert_skroutz_smart_cart_special_id',false));
		        $color_parent_attributes=[];
		        if ($special_id) {
			        $try_to_get_parent = explode("-", $line_item->shop_uid);
			        if (sizeof($try_to_get_parent)>1) {
				        $n = sizeof($try_to_get_parent);
				        for ($i = 0; $i < $n; $i++) {
					        $try_to_get_parent[$i] = implode("-", isset($try_to_get_parent[$i-1]) ? [$try_to_get_parent[$i-1], $try_to_get_parent[$i]] : [$try_to_get_parent[$i]]);
					        $product_parent = $try_to_get_parent[$i];
					        if (get_option('webexpert_skroutz_smart_cart_unique_id','sku')=='sku') {
						        $product_id = wc_get_product_id_by_sku($product_parent);
						        if ($product_id) {
							        $product=wc_get_product($product_id);
						        }else {
							        $product = wc_get_product(apply_filters('webexpert_smart_cart_custom_product_if_sku',null,$line_item));
						        }
					        }else {
						        $product=wc_get_product($product_parent);
					        }
					        if ($product) {
						        for ($z=1;$z<sizeof($try_to_get_parent)-$i;$z++) {
							        $term=get_term($try_to_get_parent[$z+$i]);
							        if (!is_wp_error($term)) {
								        $color_parent_attributes["attribute_{$term->taxonomy}"]=$term->slug;
							        }
						        }
						        break;
					        }
				        }
			        }else{
				        if (get_option('webexpert_skroutz_smart_cart_unique_id','sku')=='sku') {
					        $product_id = wc_get_product_id_by_sku($line_item->shop_uid);
					        if ($product_id) {
						        $product=wc_get_product($product_id);
					        }else {
						        $product = wc_get_product(apply_filters('webexpert_smart_cart_custom_product_if_sku',null,$line_item));
					        }
				        }else {
					        $custom_uid=get_option('we_skroutz_xml_custom_id',null);
					        if ($custom_uid=='_sku') {
						        $product=wc_get_product_id_by_sku($line_item->shop_uid);
					        }else {
						        if (apply_filters('webexpert_smart_cart_custom_uid',$custom_uid,$line_item)) {
							        $meta_key=apply_filters('webexpert_smart_cart_custom_uid',$custom_uid,$line_item);
							        $args = array(
								        'type' => 'simple',
								        'post_type' => 'product',
								        'post_status' => 'publish',
								        'meta_key' => $meta_key,
								        'meta_value' => $line_item->shop_uid,
								        'meta_compare' => 'IN'
							        );
							        $products = wc_get_products($args);
							        if (is_array($products) && !empty($products)) {
								        $product=reset($products);
							        }else {
								        $product = wc_get_product($line_item->shop_variation_uid ?? $line_item->shop_uid);
							        }
						        }else {
							        $product=wc_get_product($line_item->shop_variation_uid ?? $line_item->shop_uid);
						        }
					        }
				        }
			        }
		        }else {
			        $product=wc_get_product($line_item->shop_variation_uid ?? $line_item->shop_uid);
		        }

		        if (!$product) {
			        $product = wc_get_product(apply_filters('webexpert_smart_cart_custom_product_if_sku',null,$line_item->shop_uid));
		        }

		        if (!$product) {
			        if (get_option('webexpert_skroutz_smart_cart_unique_id','sku')=='sku') {
				        $product_id = wc_get_product_id_by_sku($line_item->shop_uid);
				        if ($product_id) {
					        $product=wc_get_product($product_id);
				        }
			        }else {
				        $custom_uid=get_option('we_skroutz_xml_custom_id',null);
				        if ($custom_uid=='_sku') {
					        $product=wc_get_product_id_by_sku($line_item->shop_uid);
				        }else {
					        if (apply_filters('webexpert_smart_cart_custom_uid',$custom_uid,$line_item)) {
						        $meta_key=apply_filters('webexpert_smart_cart_custom_uid',$custom_uid, $line_item);
						        $args = array(
							        'type' => 'simple',
							        'post_type' => 'product',
							        'post_status' => 'publish',
							        'meta_key' => $meta_key,
							        'meta_value' => $line_item->shop_uid,
							        'meta_compare' => 'IN'
						        );
						        $products = wc_get_products($args);
						        if (is_array($products) && !empty($products)) {
							        $product=reset($products);
						        }else {
							        $product = wc_get_product($line_item->shop_variation_uid ?? $line_item->shop_uid);
						        }
					        }else {
						        $product=wc_get_product($line_item->shop_variation_uid ?? $line_item->shop_uid);
					        }
				        }
			        }
		        }

		        if ($product) {
			        $item_already_added = false;
			        foreach ( $order->get_items() as $order_item ) {
				        if ( $order_item->get_meta( 'item_id' ) == $line_item->id ) {
					        $item_already_added = true;
					        break;
				        }
			        }

			        if ($product->is_type('variable')) {
				        $variation=null;
				        $variation_id=null;
				        if (!empty($line_item->shop_variation_uid)) {
					        $custom_uid=get_option('we_skroutz_xml_custom_id',null);
					        if (!empty($custom_uid)) {
						        if ($custom_uid == "_sku") {
							        $variation=wc_get_product(wc_get_product_id_by_sku($line_item->shop_variation_uid));
							        if ($variation) {
								        $variation_id=$variation->get_id();
							        }
						        }else {
							        $posts = get_posts(array(
								        'post_type'      => 'prodct',
								        'post_status'    => 'publish',
								        'meta_key'       => $custom_uid,
								        'meta_value'     => $line_item->shop_variation_uid,
								        'numberposts'    => 1,
								        'fields'=>'ids'
							        ));

							        if (!empty($posts)) {
								        $variation=wc_get_product(reset($posts));
										if ($variation) {
											$variation_id=$variation->get_id();
										}
							        }
						        }
					        }else {
						        $variation=wc_get_product($line_item->shop_variation_uid);
						        if ($variation) {
							        $variation_id=$variation->get_id();
						        }
					        }
				        }

				        if (empty($variation_id) && isset($line_item->size) && (!empty($line_item->size->shop_value) || !empty($line_item->size->value))) {
					        $shop_value=!empty($line_item->size->shop_value) ? $line_item->size->shop_value : $line_item->size->value;
					        $variation_attributes=get_option('webexpert_skroutz_smart_cart_variation_attributes',[]);
					        foreach ($variation_attributes as $variation_attribute) {
						        // trying to obtain term
						        $t=get_term_by('name',$shop_value,wc_attribute_taxonomy_name($variation_attribute));
						        if (!is_wp_error($t)) {
							        $variation_id=$this->webex_find_variation($product->get_id(),array_merge($color_parent_attributes,['attribute_pa_'.$variation_attribute=>$t->slug]));
							        if (!empty($variation_id)) {
								        break;
							        }
						        }
					        }
				        }
			        }

			        if ( !$item_already_added ) {
				        $args = [
					        'name'     => $line_item->product_name,
					        'subtotal' => $line_item->unit_price * $line_item->quantity,
					        'total'    => $line_item->total_price,
					        'quantity' => $line_item->quantity,
				        ];

				        if ( wc_tax_enabled() && $line_item->price_includes_vat) {
					        $vat_rate = $line_item->vat_ratio; // e.g. 24
					        $quantity = $line_item->quantity;
					        $net_unit_price = $line_item->unit_price / (1 + ($vat_rate / 100));
					        $net_total = $net_unit_price * $quantity;

					        $args = [
						        'name'     => $line_item->product_name,
						        'subtotal' => wc_format_decimal($net_unit_price * $quantity, 4),
						        'total'    => wc_format_decimal($net_total, 4),
						        'quantity' => $quantity,
					        ];
				        }

				        if ($product->is_type('variable') && !empty($variation_id)) {
					        $args['variation_id']    = $variation_id;
				        }

				        $line_item_id = $order->add_product( $product, $line_item->quantity, $args );
				        $order_item   = $order->get_item( $line_item_id );
				        if ( $order_item instanceof WC_Order_Item_Product ) {
					        if ( ! empty( $line_item->extra_info ) ) {
						        $order_item->update_meta_data( apply_filters( 'webexpert_smart_cart_custom_extra_info_field', 'extra_info' ), $line_item->extra_info );
					        }
					        $order_item->update_meta_data( 'item_id', $line_item->id, true );
					        $order_item->save();
				        }

				        if ( !empty($line_item->island_vat_discount_applied) && get_option('webexpert_timologio_for_woocommerce_enable_island_reduced_tax','')=='yes') {
					        $newTax = null;
					        if ($line_item->vat_ratio == 17.0) {
						        $newTax = get_option('webexpert_timologio_for_woocommerce_island_reduced_tax_class_for_24', 'reduced-24');
					        } elseif ($line_item->vat_ratio == 13.0) {
						        $newTax = get_option('webexpert_timologio_for_woocommerce_island_reduced_tax_class_for_13', 'reduced-13');
					        } elseif ($line_item->vat_ratio == 6.0) {
						        $newTax = get_option('webexpert_timologio_for_woocommerce_island_reduced_tax_class_for_6', 'reduced-6');
					        }
					        $order->update_meta_data('_reduced_vat_for_island', 'true');
					        if (!empty($newTax)) {
						        $order_item->set_tax_class( $newTax );
					        }
					        $order_item->save();
				        }
			        }
		        }
	        }

	        if (isset($sc_order->invoice) && $sc_order->invoice=='true') {
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_invoice_toggle','_billing_invoice'), apply_filters('webexpert_smart_cart_invoice_value_toggle','y'));
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_billing_activity','_billing_activity'), $sc_order->invoice_details->profession);
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_billing_vat_id','_billing_vat_id'), $sc_order->invoice_details->vat_number);
		        $order->update_meta_data(apply_filters('webexpert_smart_cart_billing_tax_office','_billing_tax_office'), $sc_order->invoice_details->doy);

				if (get_option('webexpert_skroutz_smart_cart_merge_addresses','yes')=='yes') {
					$order->set_billing_address_1("{$sc_order->invoice_details->address->street_name} {$sc_order->invoice_details->address->street_number}");
				}else {
					$order->set_billing_address_1($sc_order->invoice_details->address->street_name);
					$order->set_billing_address_2($sc_order->invoice_details->address->street_number);
				}

		        $order->set_billing_postcode($sc_order->invoice_details->address->zip);
		        $order->set_billing_city($sc_order->invoice_details->address->city);
		        $order->set_billing_company($sc_order->invoice_details->company);
	        }else {
		        $order->update_meta_data( apply_filters('webexpert_smart_cart_invoice_toggle','_billing_invoice'), 'n');
	        }

	        $shipping_items    = $order->get_items('shipping');
	        if ( sizeof( $shipping_items ) == 0 ) {
		        $separate=explode(":",$preferred_shipping_method);
		        $item = new WC_Order_Item_Shipping();

		        $methods=[];
		        $zone = new \WC_Shipping_Zone( 0 );
		        foreach ( $zone->get_shipping_methods() as $shipping_method ) {
			        $id=$shipping_method->id;
			        $id.=!empty($shipping_method->get_instance_id()) ? ':'.$shipping_method->get_instance_id() : '';
			        $methods[$id]=$shipping_method->get_method_title();
		        }

		        $zones = WC_Shipping_Zones::get_zones();
		        foreach ( $zones as $zone ) {
			        $zone = new \WC_Shipping_Zone( $zone['id'] );
			        foreach ( $zone->get_shipping_methods() as $shipping_method ) {
				        $id=$shipping_method->id;
				        $id.=!empty($shipping_method->get_instance_id()) ? ':'.$shipping_method->get_instance_id() : '';
				        $methods[$id]=$shipping_method->get_title();
			        }
		        }

		        if (is_array($separate) && sizeof($separate)==2) {
			        $item->set_method_id( $separate[0] );
			        $item->set_instance_id( $separate[1] );
		        }else {
			        $item->set_method_id( $preferred_shipping_method );
		        }
		        $item->set_method_title($methods[$preferred_shipping_method]);
		        $order->add_item( $item );
	        }
	        $order->calculate_totals();
	        $order_id=$order->save();
	        $wp_filesystem->delete($lock_file);

            if ($order_id) {
	            do_action( 'webexpert_skroutz_smart_cart_order_inserted', $order_id );
            	return $order_id;
            }
            return false;
        }catch (Exception $e) {
            echo $e->getMessage();
	        $wp_filesystem->delete($lock_file);
	        return false;
        }
    }

    function webexpert_skroutz_smart_cart_register_for_return_status($order_statuses) {
	    $order_statuses['wc-for-return'] = array(
		    'label'                     => _x( 'For return', 'Order status', 'webexpert-woocommerce-skroutz-smart-cart' ),
		    'public'                    => false,
		    'exclude_from_search'       => false,
		    'show_in_admin_all_list'    => true,
		    'show_in_admin_status_list' => true,
		    'label_count'               => _n_noop( 'For return <span class="count">(%s)</span>', 'For return <span class="count">(%s)</span>', 'webexpert-woocommerce-skroutz-smart-cart' ),
	    );
	    return $order_statuses;
    }

    function webexpert_skroutz_smart_cart_show_for_return_status($order_statuses) {
	    $order_statuses['wc-for-return'] = _x( 'For return', 'Order status', 'webexpert-woocommerce-skroutz-smart-cart' );
	    return $order_statuses;
    }

    function webexpert_skroutz_smart_cart_get_for_return_status_bulk($bulk_actions) {
	    $bulk_actions['mark_for-return'] = 'Change status to custom status';
	    return $bulk_actions;
    }

	function webexpert_webexpert_skroutz_smart_cart_meta_box($post_type, $post) {
		if (!in_array($post_type, ["shop_order",'woocommerce_page_wc-orders'])) {
			return;
		}

		$order = ( $post instanceof WP_Post ) ? wc_get_order( $post->ID ) : $post;
		if (strpos($order->get_order_key(), 'SC-') !== false) {
			$screen = wc_get_container()->get( \Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class )->custom_orders_table_usage_is_enabled()
				? wc_get_page_screen_id( 'shop-order' )
				: 'shop_order';

			add_meta_box(
				'webexpert_skroutz_smart_cart',
				__('Skroutz Smart Cart', 'webexpert-woocommerce-order-tracking'),
				[$this, 'webexpert_skroutz_smart_cart_metabox_content'],
				$screen, 'side', 'core'
			);
		}
	}

	function webexpert_skroutz_smart_cart_metabox_content($post) {
		$order = ( $post instanceof WP_Post ) ? wc_get_order( $post->ID ) : $post;
		if ($order->get_created_via()=='skroutz_smart_cart') {
            $store_pickup=!empty($order->get_meta('_skroutz_store_pickup'));
			echo "<div class='wrapper'>";
			$order_key=str_replace("SC-","",$order->get_order_key());
			$response=$this->fetch('orders/'.$order_key);
			if ($response->order->state=='open') {
				echo '<div class="webexpert-skroutz-status open"><span>'.__('New','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='accepted') {
				echo '<div class="webexpert-skroutz-status accepted"><span>'.__('To be sent','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='rejected') {
				echo '<div class="webexpert-skroutz-status rejected"><span>'.__('Rejected','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='dispatched') {
				echo '<div class="webexpert-skroutz-status dispatched"><span>'.__('Dispatched','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='delivered') {
				echo '<div class="webexpert-skroutz-status delivered"><span>'.__('Delivered','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='expired') {
				echo '<div class="webexpert-skroutz-status expired"><span>'.__('Expired','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='cancelled') {
				echo '<div class="webexpert-skroutz-status cancelled"><span>'.__('Cancelled','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='returned') {
				echo '<div class="webexpert-skroutz-status returned"><span>'.__('Returned','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='partially_returned') {
				echo '<div class="webexpert-skroutz-status partially_returned"><span>'.__('Partially Returned','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}elseif ($response->order->state=='for_return') {
				echo '<div class="webexpert-skroutz-status for_return"><span>'.__('For return','webexpert-woocommerce-skroutz-smart-cart').'</span></div>';
			}

			echo "<div class='webexpert-field-label'>".sprintf(_x('Order: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key)."</div>";
			if ($response->order->courier_voucher) {
				echo "<div class='webexpert-field-label'>".__('Carrier','webexpert-woocommerce-skroutz-smart-cart').": ".$response->order->courier."</div>";
				echo "<div class='webexpert-field-label'>".__('Voucher','webexpert-woocommerce-skroutz-smart-cart').": <a target='_blank' href='".$response->order->courier_voucher."'>".reset($response->order->courier_tracking_codes)."</a></div>";
			}

            if ($response->order->store_pickup || $store_pickup) {
                echo "<div class='webexpert-field-label'>".__('Store pickup','webexpert-woocommerce-skroutz-smart-cart').": ".__('Yes','webexpert-woocommerce-skroutz-smart-cart')."</div>";
            }
            if ($response->order->gift_wrap || !empty($order->get_meta('gift_wrap'))) {
                echo "<div class='webexpert-field-label'>".__('Gift wrap','webexpert-woocommerce-skroutz-smart-cart').": ".__('Yes','webexpert-woocommerce-skroutz-smart-cart')."</div>";
            }

			echo "<div style='clear:both'></div>";

			if (!$store_pickup && $response->order->state=='open') {
				echo '<div class="webexpert-field-label">';
				echo '<label for="smart_cart_order_pickup_location">'.__('Pickup location','webexpert-woocommerce-skroutz-smart-cart').'</label>';
				echo "<select id='smart_cart_order_pickup_location'>";
				foreach ($response->order->accept_options->pickup_location as $pickup_location) {
					echo "<option value='{$pickup_location->id}'>{$pickup_location->label}</option>";
				}
				echo "</select>";
				echo "</div>";

				echo '<div class="webexpert-field-label">';
				echo '<label for="smart_cart_order_pickup_window">'.__('Pickup window','webexpert-woocommerce-skroutz-smart-cart').'</label>';
				echo "<select id='smart_cart_order_pickup_window'>";
				foreach ($response->order->accept_options->pickup_window as $pickup_window) {
					echo "<option value='{$pickup_window->id}'>{$pickup_window->label}</option>";
				}
				echo "</select>";
				echo "</div>";
			}

			echo "<div class='webexpert-field-label'>";
			if (!$store_pickup && $response->order->state=='open') {
				echo "<button type='button' class='button accept-skroutz has-spinner' id='smart_cart_accept_order' data-order='" . $order_key . "'>" . __('Accept', 'webexpert-woocommerce-skroutz-smart-cart') . "</button>";
			}
			if (!$store_pickup && ($response->order->state=='open' || $response->order->state=='accepted')) {
				echo "<button type='button' class='button reject-skroutz' id='smart_cart_reject_toggle'>" . __('Reject', 'webexpert-woocommerce-skroutz-smart-cart') . "</button>";
			}
			echo "<div style='clear:both'></div>";

			echo "</div>";
			echo "<div style='clear:both'></div>";
			if (!$store_pickup && ($response->order->state=='open' || $response->order->state=='accepted')) {
				echo "<div id='smart_cart_order_reject_container'>";
				if ($response->order->state=='open') {
					echo "<div class='webexpert-field-label'>";
					echo "<select id='smart_cart_order_reject_reason'>";
					foreach ($response->order->reject_options->line_item_rejection_reasons as $line_item_rejection_reason) {
						echo "<option value='{$line_item_rejection_reason->id}'>{$line_item_rejection_reason->label}</option>";
					}
					echo "</select>";
					echo "</div>";

				}
				echo "<div class='webexpert-field-label'>";
				echo "<input type='text' placeholder='".__('Other reason','webexpert-woocommerce-skroutz-smart-cart')."' id='smart_cart_order_reject_reason_other'>";
				echo "<input type='hidden' id='smart_cart_order_line_numbers' value='".implode(',', array_column($response->order->line_items, 'id'))."'>";
				echo "</div>";
				if ($response->order->state=='open' || $response->order->state=='accepted') {
					echo "<button style='float:left' type='button' class='button reject-skroutz has-spinner' id='smart_cart_reject_order' data-order='" . $order_key . "'>" . __('Reject', 'webexpert-woocommerce-skroutz-smart-cart') . "</button>";
				}
				echo "<div style='clear:both'></div>";
				echo "</div>";
			}
			echo "</div>";
		}
	}

    function process_order( $response ) {
        $body = file_get_contents('php://input');
        $response = json_decode($body);

	    if (get_option('webexpert_skroutz_smart_cart_debug',null)=='yes') {
		    $logger = wc_get_logger();
		    $logger->info( wc_print_r( json_encode($response,JSON_UNESCAPED_UNICODE), true ), array( 'source' => 'smart-cart' ) );
        }

	    do_action('webexpert_skroutz_smart_cart_webhook_triggered',$response);
	    $this->create_new_order($response->order,!($response->event_type == "new_order" && $response->order->state == "open"));
    }

    function webexpert_skroutz_smart_cart_grab_order() {
	    $order_key=sanitize_text_field($_POST['order_id']);
	    $response=$this->fetch('orders/'.$order_key);
	    if ($response && isset($response->order)) {
		    if (get_option('webexpert_skroutz_smart_cart_debug',null)=='yes') {
			    $logger = wc_get_logger();
			    $logger->info( wc_print_r( json_encode($response,JSON_UNESCAPED_UNICODE), true ), array( 'source' => 'smart-cart' ) );
		    }

		    $order_id=$this->create_new_order($response->order,true, true);
		    if ($order_id) {
			    wp_send_json_success(['order_id'=>$order_id, 'order_url' => admin_url('post.php?post=' . $order_id . '&action=edit')]);
		    }else {
			    wp_send_json_error();
		    }
	    }
	    wp_send_json(['success'=>0]);
    }

    function webexpert_skroutz_smart_cart_accept_order() {
	    $order_key=sanitize_text_field($_POST['order_id']);
	    $pickup_location=sanitize_text_field($_POST['pickup_location']);
	    $pickup_window=sanitize_text_field($_POST['pickup_window']);
	    $number_of_parcels=sanitize_text_field($_POST['number_of_parcels']);

	    $order_id=wc_get_order_id_by_order_key("SC-$order_key");
	    if ($order_id) {
	    	$order=wc_get_order($order_id);
		    $args=[
			    'number_of_parcels'=>$number_of_parcels,
			    'pickup_location'=>$pickup_location,
			    'pickup_window'=>$pickup_window,
		    ];

		    $response=$this->fetch('orders/'.$order_key.'/accept','POST',$args);

		    if (isset($response->errors) && sizeof($response->errors)>0) {
			    $errors=implode(',', array_map(function($x) { return implode(', ',$x->messages); }, $response->errors));
			    $note = sprintf(_x('Order %s has failed to be accepted on Skroutz smart cart. Errors: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key, $errors);
			    $order->add_order_note( $note );
			    $order->save();
			    wp_send_json(['success'=>0,'errors'=>$errors]);
		    }

		    if ($response->success) {
			    $note = sprintf(_x('Order %s has been approved at Skroutz smart cart.', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key);
			    $order->add_order_note( $note );
			    $order->save();
			    $order->update_status('processing');
			    wp_send_json(['success'=>1,'errors'=>null]);
		    }
	    }
    }

    function webexpert_skroutz_smart_cart_reject_order() {

	    $order_key=sanitize_text_field($_POST['order_id']);
	    $reject_reason=sanitize_text_field($_POST['reject_reason']);
	    $reject_reason_other=sanitize_text_field($_POST['reject_reason_other']);
	    $line_numbers=sanitize_text_field($_POST['line_numbers']);

	    $order_id=wc_get_order_id_by_order_key("SC-$order_key");
	    if ($order_id) {
		    $order = wc_get_order($order_id);

		    if (!empty($reject_reason_other)) {
			    $args['rejection_reason_other'] = $reject_reason_other;
		    }else {
			    $line_items=explode(",",$line_numbers);
			    $args['line_items'] = [];

			    foreach ($line_items as $line_item) {
				    array_push($args['line_items'],['id'=>$line_item,'reason'=>$reject_reason]);
			    }
		    }

		    $response = $this->fetch('orders/' . $order_key . '/reject', 'POST', $args);
		    if ($response->success) {
			    $note = sprintf(_x('Order %s has been rejected at Skroutz smart cart with reason: '.$reject_reason_other, 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key);
			    $order->add_order_note( $note );
			    $order->save();
			    $order->update_status('cancelled');
			    wp_send_json(['success'=>1,'errors'=>null]);
		    }

		    if (isset($response->errors) && sizeof($response->errors)>0) {
			    $errors=implode(',', array_map(function($x) { return implode(', ',$x->messages); }, $response->errors));
			    $note = sprintf(_x('Order %s has failed to be rejected on Skroutz smart cart. Errors: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key, $errors);
			    $order->add_order_note( $note );
			    $order->save();
			    wp_send_json(['success'=>0,'errors'=>$errors]);
		    }
	    }
    }

	function on_hold_to_processing($order_id,$old_status,$new_status) {
		if (get_option('webexpert_skroutz_smart_cart_manage_orders', null)!="yes")
			return;

		$order=wc_get_order($order_id);
		if ($order) {
			if ($order->get_created_via()=='skroutz_smart_cart') {
				$order_key=str_replace("SC-","",$order->get_order_key());
				$response=$this->fetch('orders/'.$order_key);

				if ($old_status=="on-hold" && $new_status=="processing") {
					if ($response->order->state=='open') {
						if (get_option('webexpert_skroutz_smart_cart_auto_accept_orders_pickup_window','first_pickup_window')=="last_pickup_window") {
							$args=[
								'number_of_parcels'=>reset($response->order->accept_options->number_of_parcels),
								'pickup_location'=>end($response->order->accept_options->pickup_location)->id,
								'pickup_window'=>end($response->order->accept_options->pickup_window)->id,
							];
						}else {
							$args=[
								'number_of_parcels'=>reset($response->order->accept_options->number_of_parcels),
								'pickup_location'=>reset($response->order->accept_options->pickup_location)->id,
								'pickup_window'=>reset($response->order->accept_options->pickup_window)->id,
							];
						}

						$response=$this->fetch('orders/'.$order_key.'/accept','POST',$args);

						if (isset($response->errors) && sizeof($response->errors)>0) {
							$errors=implode(',', array_map(function($x) { return implode(', ',$x->messages); }, $response->errors));
							$note = sprintf(_x('Order %s has failed to be accepted on Skroutz smart cart. Errors: %s', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key, $errors);
							$order->add_order_note( $note );
							$order->save();
							$order->update_status('on-hold');
						}

						if ($response->success) {
							$note = sprintf(_x('Order %s has been approved at Skroutz smart cart.', 'the order key', 'webexpert-woocommerce-skroutz-smart-cart'), $order_key);
							$order->add_order_note( $note );
							$order->save();
						}
					}
				}
			}
		}
	}

	function auto_accept_order($order_id) {
		if (get_option('webexpert_skroutz_smart_cart_auto_accept_orders',null)=='yes') {
			$order=wc_get_order($order_id);
			if ($order && $order->get_created_via()=='skroutz_smart_cart' && $order->get_status()=='on-hold') {
                if (!empty($order->get_meta('_skroutz_store_pickup'))) {
                    return;
                }

                $order_key=str_replace("SC-","",$order->get_order_key());
                $response=$this->fetch('orders/'.$order_key);
                if (!empty($response->order->line_items) && !empty($order->get_items()) && count($response->order->line_items) == count( $order->get_items() )) {
	                $order->update_status('processing');
                }
			}
		}
	}

	function redirect_mails($args){
		if (is_array($args['to'])) {
			$args['to'] = array_filter($args['to'], function($email) {
				$domain_name = substr(strrchr($email, "@"), 1);
				return $domain_name !== "skroutz.gr";
			});
		} else {
			$domain_name = substr(strrchr($args['to'], "@"), 1);
			if ($domain_name === "skroutz.gr") {
				$args['to'] = '';
			}
		}
		return $args;
	}

	function webex_find_variation($product_id, $attributes) {
		$product_data_store = new WC_Product_Data_Store_CPT();
		$product = new WC_Product($product_id);

		return $product_data_store->find_matching_product_variation($product, $attributes);
	}
}