<?php
/**
 * Plugin Name: WooCommerce Auto Greeklish Slugs
 * Description: Αυτόματη μετατροπή Ελληνικών σε Google-friendly Greeklish ΜΟΝΟ στα URL του WooCommerce (προϊόντα, κατηγορίες, ετικέτες, ιδιότητες & όρους ιδιοτήτων).
 * Version: 1.6.0
 * Author: ProMarketing.gr
 */

if (!defined('ABSPATH')) exit;

/* ======================================================
   SETTINGS
====================================================== */
function wcags_enabled() {
    return get_option('wcags_enabled', 'yes') === 'yes';
}

/* ======================================================
   ΕΛΛΗΝΙΚΑ → GREEKLISH
====================================================== */
function wcags_el_to_greeklish($string) {

    $map = [
        'α'=>'a','ά'=>'a','β'=>'v','γ'=>'g','δ'=>'d','ε'=>'e','έ'=>'e',
        'ζ'=>'z','η'=>'i','ή'=>'i','θ'=>'th','ι'=>'i','ί'=>'i','ϊ'=>'i','ΐ'=>'i',
        'κ'=>'k','λ'=>'l','μ'=>'m','ν'=>'n','ξ'=>'x','ο'=>'o','ό'=>'o','π'=>'p',
        'ρ'=>'r','σ'=>'s','ς'=>'s','τ'=>'t','υ'=>'y','ύ'=>'y','ϋ'=>'y','ΰ'=>'y',
        'φ'=>'f','χ'=>'ch','ψ'=>'ps','ω'=>'o','ώ'=>'o'
    ];

    $string = mb_strtolower($string, 'UTF-8');
    $string = strtr($string, $map);
    $string = preg_replace('/[^a-z0-9\s-]/', '', $string);
    $string = preg_replace('/\s+/', '-', $string);
    $string = preg_replace('/-+/', '-', $string);

    return trim($string, '-');
}

/* ======================================================
   PRODUCTS
====================================================== */
add_filter('wp_unique_post_slug', function ($slug, $post_id, $status, $type) {

    if (!wcags_enabled()) return $slug;
    if ($type !== 'product') return $slug;

    $title = get_post_field('post_title', $post_id);
    if (!$title) return $slug;

    return wcags_el_to_greeklish($title);

}, 10, 4);

/* ======================================================
   PRODUCT CATEGORIES & TAGS
====================================================== */
add_action('created_term', 'wcags_fix_term_slug', 20, 3);
add_action('edited_term',  'wcags_fix_term_slug', 20, 3);

function wcags_fix_term_slug($term_id, $tt_id, $taxonomy) {

    if (!wcags_enabled()) return;

    // product categories & tags
    if (in_array($taxonomy, ['product_cat','product_tag'])) {
        wcags_update_term_slug($term_id, $taxonomy);
    }

    // attribute terms (pa_*)
    if (strpos($taxonomy, 'pa_') === 0) {
        wcags_update_term_slug($term_id, $taxonomy);
    }
}

/* ======================================================
   TERM SLUG UPDATE (SAFE, REUSABLE)
====================================================== */
function wcags_update_term_slug($term_id, $taxonomy) {

    if (get_term_meta($term_id, '_wcags_done', true)) return;

    $term = get_term($term_id, $taxonomy);
    if (!$term || is_wp_error($term)) return;

    $new_slug = wcags_el_to_greeklish($term->name);

    if ($term->slug !== $new_slug && $new_slug !== '') {
        wp_update_term($term_id, $taxonomy, ['slug' => $new_slug]);
    }

    update_term_meta($term_id, '_wcags_done', 1);
}

/* ======================================================
   ATTRIBUTES (DEFINITION)
====================================================== */
add_action('woocommerce_attribute_added',   'wcags_fix_attribute_slug', 10, 2);
add_action('woocommerce_attribute_updated', 'wcags_fix_attribute_slug', 10, 2);

function wcags_fix_attribute_slug($attribute_id, $data) {

    if (!wcags_enabled()) return;
    if (empty($data['attribute_name'])) return;

    global $wpdb;

    $new_slug = wcags_el_to_greeklish($data['attribute_name']);

    $table = $wpdb->prefix . 'woocommerce_attribute_taxonomies';

    $wpdb->update(
        $table,
        ['attribute_name' => $new_slug],
        ['attribute_id'   => $attribute_id]
    );

    delete_transient('wc_attribute_taxonomies');
    wc_delete_product_transients();
}

/* ======================================================
   ADMIN SETTINGS
====================================================== */
add_action('admin_menu', function () {
    add_submenu_page(
        'woocommerce',
        'Auto Greeklish Slugs',
        'Auto Greeklish Slugs',
        'manage_options',
        'wcags-settings',
        'wcags_settings_page'
    );
});

add_action('admin_init', function () {
    register_setting('wcags_settings', 'wcags_enabled');
});

function wcags_settings_page() {
?>
<div class="wrap">
    <h1>WooCommerce Auto Greeklish Slugs</h1>

    <form method="post" action="options.php">
        <?php settings_fields('wcags_settings'); ?>
        <table class="form-table">
            <tr>
                <th>Ενεργοποίηση</th>
                <td>
                    <select name="wcags_enabled">
                        <option value="yes" <?php selected(get_option('wcags_enabled','yes'),'yes'); ?>>ON</option>
                        <option value="no"  <?php selected(get_option('wcags_enabled','yes'),'no'); ?>>OFF</option>
                    </select>
                </td>
            </tr>
        </table>
        <?php submit_button(); ?>
    </form>
</div>
<?php
}
