<?php
/**
 * Custom WooCommerce Email: Invoice/Receipt Notification.
 *
 * This class defines a custom email that can be sent to the customer with
 * an invoice, receipt, or other document related to the order.
 *
 * @class       WC_Email_Invoice_Email
 * @extends     WC_Email
 * @package     Oxygen\Emails
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class WC_Email_Invoice_Email
 *
 * Handles sending custom invoice/receipt email notifications to customers.
 */
class WC_Email_Invoice_Email extends WC_Email {

	/**
	 * Constructor for the custom invoice email class.
	 *
	 * Sets the email ID, title, subject, description, templates, and other
	 * required properties. Also defines the path to the plugin's templates.
	 */
	public function __construct() {
		$this->id          = 'invoice_email';
		$this->title       = __( 'Invoice Email', 'oxygen' );
		$this->description = __( 'You can use this type of email in order to send the document receipt/invoice of the order.', 'oxygen' );

		$this->heading = __( 'Document invoice/receipt/notice for this order.', 'oxygen' );
		$this->subject = __( 'Document invoice/receipt/notice for this order.', 'oxygen' );

		// Point to the template within the plugin.
		$this->template_html  = 'templates/invoice-email-template.php';
		$this->template_plain = 'templates/plain-invoice-email-template.php';

		$this->customer_email = true;

		// Specify the plugin directory where templates are located.
		$this->template_base = plugin_dir_path( __FILE__ );

		parent::__construct();
	}

	/**
	 * Get the HTML content for the email.
	 *
	 * @param int $order_id WC_Order id.
	 * @return string The HTML content of the email.
	 */
	public function trigger( $order_id ) {
		if ( ! $order_id ) {
			return;
		}

		$this->object    = wc_get_order( $order_id );
		$this->recipient = $this->object->get_billing_email();

		if ( ! $this->is_enabled() || ! $this->get_recipient() ) {
			return;
		}

		$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
	}

	/**
	 * Get the HTML content for the email.
	 *
	 * @return string The HTML content of the email.
	 */
	public function get_content_html() {
		return wc_get_template_html(
			$this->template_html,
			array(
				'order'         => $this->object,
				'email_heading' => $this->get_heading(),
				'sent_to_admin' => false,
				'plain_text'    => false,
				'email'         => $this,
				'recipient'     => $this->recipient,
			),
			'', // No override by theme.
			$this->template_base
		);
	}

	/**
	 * Get the plain text content for the email.
	 *
	 * @return string The plain text content of the email.
	 */
	public function get_content_plain() {
		return wc_get_template_html(
			$this->template_plain,
			array(
				'order'         => $this->object,
				'email_heading' => $this->get_heading(),
				'sent_to_admin' => false,
				'plain_text'    => true,
				'email'         => $this,
				'recipient'     => $this->recipient,
			),
			'', // No override by theme.
			$this->template_base
		);
	}
}
