<?php
/**
 * Class CheckBeforeActivation
 *
 * Ensures that the plugin is only activated if WooCommerce is installed and active.
 *
 * Responsibilities:
 * - Checks if WooCommerce is active during plugin activation.
 * - If not active, sets a flag to show an admin notice and prevents further execution.
 * - Displays an admin notice informing the user that WooCommerce is required.
 *
 * Usage:
 * - Call `CheckBeforeActivation::oxygen_woo_plugin_check_activation( __FILE__ )` during plugin activation.
 * - Hook `CheckBeforeActivation::oxygen_woo_plugin_activation_admin_notice` to `admin_notices` to display the message.
 *
 * @package Oxygen_MyData_Plugin
 */

defined( 'ABSPATH' ) || exit;

/**
 * Class CheckBeforeActivation
 */
class CheckBeforeActivation {

	/**
	 * Checks if WooCommerce is active during plugin activation.
	 * If not, sets a flag to trigger an admin notice.
	 *
	 * @param string $main_plugin_file Path to the main plugin file.
	 */
	public static function oxygen_woo_plugin_check_activation( $main_plugin_file ) {

		if ( ! class_exists( 'OxygenHelperFunctions' ) ) {
			require_once OXYGEN_PLUGIN_DIR . '/inc/class-oxygenhelperfunctions.php';

			OxygenHelperFunctions::debug( array( 'in oxygen_woo_plugin_check_activation' ) );

			if ( ! function_exists( 'is_plugin_active' ) ) {
				include_once ABSPATH . 'wp-admin/includes/plugin.php';
			}

			if ( ! is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
				OxygenHelperFunctions::debug(
					array( 'Plugin file to deactivate: ' . $main_plugin_file )
				);

				update_option( 'oxygen_woo_plugin_activation_error', true );
			}
		}
	}

	/**
	 * Displays an admin notice if WooCommerce is not active.
	 *
	 * Hook this to admin_notices.
	 */
	public static function oxygen_woo_plugin_activation_admin_notice() {
		if ( get_option( 'oxygen_woo_plugin_activation_error' ) ) {
			echo '<div class="notice notice-error is-dismissible">';
			echo '<p>' . esc_html__( 'This plugin requires WooCommerce to be installed and active.', 'oxygen-mydata-plugin' ) . '</p>';
			echo '</div>';

			if ( ! class_exists( 'OxygenHelperFunctions' ) ) {
				require_once OXYGEN_PLUGIN_DIR . '/inc/class-oxygenhelperfunctions.php';
				OxygenHelperFunctions::debug(
					array( 'Oxygen MyData Plugin requires WooCommerce to be installed and active.' )
				);
			}

			delete_option( 'oxygen_woo_plugin_activation_error' );
		}
	}
}
