<?php
/**
 * Handles execution of Ajax actions and rendering of JSON
 */
class Loco_mvc_AjaxRouter extends Loco_hooks_Hookable {
    
    /**
     * Current ajax controller
     */
    private ?Loco_mvc_AjaxController $ctrl = null;

    /**
     * Buffer for collecting unintentional output
     */
    private ?Loco_output_Buffer $buffer = null;

    /**
     * Generate a GET request URL containing required routing parameters
     * @param string $route
     * @param array $args
     * @return string
     */
    public static function generate( string $route, array $args = [] ){
        // validate route autoload if debugging
        if( loco_debugging() && ! class_exists( self::routeToClass($route) ) ){
            throw new Exception('Loco class not found for '.$route);
        }
        $args +=  [
            'route' => $route,
            'action' => 'loco_ajax',
            'loco-nonce' => wp_create_nonce($route),
        ];
        return admin_url('admin-ajax.php','relative').'?'.http_build_query($args);
    }


    /**
     * Create a new ajax router and starts buffering output immediately
     */
    public function __construct(){
        $this->buffer = Loco_output_Buffer::start();
        parent::__construct();
    }


    /**
     * "init" action callback.
     * early-ish hook that ensures controllers can initialize
     */
    public function on_init(){
        $this->ctrl = null;
        $class = self::routeToClass( $_REQUEST['route'] );
        if( class_exists($class) ){
            $this->ctrl = new $class;
            $this->ctrl->_init( $_REQUEST );
            // hook name compatible with AdminRouter, plus additional action for ajax hooks to set up
            do_action('loco_admin_init', $this->ctrl );
            do_action('loco_ajax_init', $this->ctrl );
        }
    }

    
    private static function routeToClass( string $route ):string {
        $route = explode( '-', $route );
        // convert route to class name, e.g. "foo-bar" => "Loco_ajax_foo_BarController"
        $key = count($route) - 1;
        $route[$key] = ucfirst( $route[$key] );
        return 'Loco_ajax_'.implode('_',$route).'Controller';
    }


    /**
     * Common ajax hook for all Loco admin JSON requests
     * Note that tests call renderAjax directly.
     * @codeCoverageIgnore
     */
    public function on_wp_ajax_loco_json(){
        $json = $this->renderAjax();
        $this->exitScript( $json,  [
            'Content-Type' => 'application/json; charset=UTF-8',
        ] );
    }


    /**
     * Additional ajax hook for download actions that won't be JSON
     * Note that tests call renderDownload directly.
     * @codeCoverageIgnore
     */
    public function on_wp_ajax_loco_download(){
        $file = null;
        $ext = null;
        $data = $this->renderDownload();
        if( is_string($data) ){
            $path = ( $this->ctrl ? $this->ctrl->get('path') : '' ) or $path = 'error.json';
            $file = new Loco_fs_File( $path );
            $ext = $file->extension();
        }
        else if( $data instanceof Exception ){
            $data = sprintf('%s in %s:%u', $data->getMessage(), basename($data->getFile()), $data->getLine() );
        }
        else {
            $data = (string) $data;
        }
        $mimes =  [
            'po'   => 'application/x-gettext',
            'pot'  => 'application/x-gettext',
            'mo'   => 'application/x-gettext-translation',
            'php'  => 'application/x-httpd-php-source',
            'json' => 'application/json',
            'zip'  => 'application/zip',
            'xml'  => 'text/xml',
        ];
        $headers = [];
        if( $file instanceof Loco_fs_File && isset($mimes[$ext]) ){
            $headers['Content-Type'] = $mimes[$ext].'; charset=UTF-8';
            $headers['Content-Disposition'] = 'attachment; filename='.$file->basename();
        }
        else {
            $headers['Content-Type'] = 'text/plain; charset=UTF-8';
        }
        $this->exitScript( $data, $headers );
    }


    /**
     * Exit script before WordPress shutdown, avoids hijacking of exit via wp_die_ajax_handler.
     * Also gives us a final chance to check for output buffering problems.
     * @codeCoverageIgnore
     */
    private function exitScript( $str, array $headers ){
        try {
            do_action('loco_admin_shutdown');
            Loco_output_Buffer::clear();
            $this->buffer = null;
            Loco_output_Buffer::check();
            $headers['Content-Length'] = strlen($str);
            foreach( $headers as $name => $value ){
                header( $name.': '.$value );
            }
        }
        catch( Exception $e ){
            Loco_error_AdminNotices::add( Loco_error_Exception::convert($e) );
            $str = $e->getMessage();
        }
        echo $str;
        exit(0);
    }


    /**
     * Execute Ajax controller to render JSON response body
     * @return string
     */
    public function renderAjax(){
        try {
            // respond with deferred failure from initAjax
            if( ! $this->ctrl ){
                $route = $_REQUEST['route'] ?? '';
                // translators: Fatal error where %s represents an unexpected value
                throw new Loco_error_Exception( sprintf( __('Ajax route not found: "%s"','loco-translate'), $route ) );
            }
            // else execute controller to get json output
            $json = $this->ctrl->render();
            if( is_null($json) || '' === $json ){
                throw new Loco_error_Exception( __('Ajax controller returned empty JSON','loco-translate') );
            }
        }
        catch( Loco_error_Exception $e ){
            $json = json_encode( [ 'error' => $e->jsonSerialize(), 'notices' => Loco_error_AdminNotices::destroy() ] );
        }
        catch( Exception $e ){
            $e = Loco_error_Exception::convert($e);
            $json = json_encode( [ 'error' => $e->jsonSerialize(), 'notices' => Loco_error_AdminNotices::destroy() ] );
        }
        $this->buffer->discard();
        return $json;
    }


    /**
     * Execute ajax controller to render something other than JSON
     * @return string|Exception
     */
    public function renderDownload(){
        try {
            // respond with deferred failure from initAjax
            if( ! $this->ctrl ){
                throw new Loco_error_Exception( __('Download action not found','loco-translate') );
            }
            // else execute controller to get raw output
            $data = $this->ctrl->render();
            if( is_null($data) || '' === $data ){
                throw new Loco_error_Exception( __('Download controller returned empty output','loco-translate') );
            }
        }
        catch( Exception $e ){
            $data = $e;
        }
        $this->buffer->discard();
        return $data;
    }

}