<?php
if ( ! defined( 'WPINC' ) ) {
	wp_die();
}


/**
 * Executed only if WooCommerce is enabled. It an extended class of Iben_Woo_Elta_Courier_Admin
 *
 * @link       https://iben.pro
 * @since      1.0.0
 *
 * @package    Iben_Woo_Elta_Courier
 * @subpackage Iben_Woo_Elta_Courier/admin
 */

/**
 *
 * @package    Iben_Woo_Elta_Courier
 * @subpackage Iben_Woo_Elta_Courier/admin
 * @author     iBen <info@ibenetos.com>
 */ 

use Automattic\WooCommerce\Utilities\OrderUtil;
use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;

if ( ! class_exists( 'Iben_Woo_Elta_Courier_Admin_Woo' ) ) {
	class Iben_Woo_Elta_Courier_Admin_Woo{
		
		private $plugin_name;
		private $version;
		private $main_settings;
		
		private $iben_elta_main_settings_enable_test_mode;
		private $iben_elta_main_settings_customer;
		private $iben_elta_main_settings_username;
		private $iben_elta_main_settings_password;
		private $iben_elta_main_settings_sub_password;
		private $iben_elta_main_settings_papersize;
		
		private $countries;

		public function __construct() {

			if ( defined( 'IBEN_WOO_ELTA_COURIER_VERSION' ) ) {
				$this->version = IBEN_WOO_ELTA_COURIER_VERSION;
			} else {
				$this->version = '1.0.0';
			}
			$this->plugin_name = 'iben-woo-elta-courier';
			
			$this->main_settings = get_option('iben_elta_main_settings_fields');

			if( isset( $this->main_settings['iben_elta_main_settings_enable_test_mode'] ) ) {
				$this->iben_elta_main_settings_enable_test_mode = sanitize_text_field( $this->main_settings['iben_elta_main_settings_enable_test_mode'] );
			}else{
				$this->iben_elta_main_settings_enable_test_mode = '';
			}

			if( isset( $this->main_settings['iben_elta_main_settings_customer'] ) ) {
				$this->iben_elta_main_settings_customer = sanitize_text_field( $this->main_settings['iben_elta_main_settings_customer'] );
			}else{
				$this->iben_elta_main_settings_customer = '';
			}

			if( isset( $this->main_settings['iben_elta_main_settings_username'] ) ) {
				$this->iben_elta_main_settings_username = sanitize_text_field( $this->main_settings['iben_elta_main_settings_username'] );
			}else{
				$this->iben_elta_main_settings_username = '';
			}

			if( isset( $this->main_settings['iben_elta_main_settings_password'] ) ) {
				$this->iben_elta_main_settings_password = sanitize_text_field( $this->main_settings['iben_elta_main_settings_password'] );
			}else{
				$this->iben_elta_main_settings_password = '';
			}
			
			
			if( isset( $this->main_settings['iben_elta_main_settings_sub_password'] ) ) {
				$this->iben_elta_main_settings_sub_password = sanitize_text_field( $this->main_settings['iben_elta_main_settings_sub_password'] );
			}else{
				$this->iben_elta_main_settings_sub_password = '';
			}

			if( isset( $this->main_settings['iben_elta_main_settings_papersize'] ) ) {
				$this->iben_elta_main_settings_papersize = sanitize_text_field( $this->main_settings['iben_elta_main_settings_papersize'] );
			}else{
				$this->iben_elta_main_settings_papersize = '';
			}

			$this->countries = array(
				'GR' => esc_html__( 'Greece', 'iben-woo-elta-courier' ),
				'CY' => esc_html__( 'Cyprus', 'iben-woo-elta-courier' )
			);

			$this->countries = apply_filters('iben_elta_filter_enabled_countries', $this->countries);


		}

		private function iben_elta_check_logins() {
			if ( empty($this->iben_elta_main_settings_customer) || empty($this->iben_elta_main_settings_username) || empty($this->iben_elta_main_settings_password) ){
				return true;
			}else{
				return false;
			}
		}


		/**
		 * Register the stylesheets for the admin area.
		 *
		 * @since    1.0.0
		 */
		public function iben_enqueue_styles() {

			// get current admin screen, or null
			$screen = get_current_screen();
			// verify admin screen object
			if (is_object($screen)) {
				// enqueue only for specific shop_order types
				if (
					in_array($screen->post_type, ['shop_order', 'nothing']) || 
					( isset( $_GET[ 'page' ]) && $_GET[ 'page' ] == 'iben_woo_elta_vouchers') ||
					( isset( $_GET[ 'page' ]) && $_GET[ 'page' ] == 'iben_woo_elta_courier')
				
				) {

					wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/iben-woo-elta-courier-admin.css', array(), $this->version, 'all' );
					
				}

			}
		}

		/**
		 * Register the JavaScript for the admin area.
		 *
		 * @since    1.0.0
		 */
		public function iben_enqueue_scripts() {


			// get current admin screen, or null
			$screen = get_current_screen();
			// verify admin screen object
			if (is_object($screen)) {
				// enqueue only for specific shop_order types
				if (
					in_array($screen->post_type, ['shop_order', 'nothing']) || 
					( isset( $_GET[ 'page' ]) && $_GET[ 'page' ] == 'iben_woo_elta_vouchers') ||
					( isset( $_GET[ 'page' ]) && $_GET[ 'page' ] == 'iben_woo_elta_courier')
				
				) {
					
					wp_enqueue_script('iben_elta_jscript', plugin_dir_url( __FILE__ ) . 'js/iben-woo-elta-courier-admin.js', array( 'jquery' ), $this->version, false );
					wp_localize_script(
						'iben_elta_jscript',
						'iben_elta_jscript',
						[
							'url' => admin_url('admin-ajax.php'),
							'iben_elta_nonce'  => wp_create_nonce( 'iben_elta_nonce' )
						]
					);

					if( isset( $this->main_settings['iben_elta_main_settings_voucher_change_order_status'] ) && $this->main_settings['iben_elta_main_settings_voucher_change_order_status'] != -1) {
						$status_text = iben_elta_get_order_status_text($this->main_settings['iben_elta_main_settings_voucher_change_order_status'] );
						$status_value = iben_elta_get_order_status_value($this->main_settings['iben_elta_main_settings_voucher_change_order_status'] );

						wp_localize_script(
							'iben_elta_jscript',
							'iben_elta_various',
							[
								'status_text' => $status_text,
								'status_value' => $status_value,
								'status_value2' => $this->main_settings['iben_elta_main_settings_voucher_change_order_status']
							]
						);	

					}

					if( isset( $this->main_settings['iben_elta_main_enable_tracking_saving'] ) ) {

						wp_localize_script(
							'iben_elta_jscript',
							'iben_elta_copy_voucher',
							[
								'tracking_enable' => 1,
								'tracking_field' => $this->main_settings['iben_elta_main_enable_tracking_saving_field']
							]
						);	

					}					

			
				}
			}

		}		
		
		/**
		 *
		 * @since     1.0.0
		 * 
		 */		
		private function ibenEltaWeServiceCreateVoucher($types = array()){

			if ( $this->iben_elta_check_logins() ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Your ELTA Courier logins required', 'iben-woo-elta-courier' );
				return $result;
			}
			if ( !extension_loaded('soap') ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Soap PHP extension must be enabled', 'iben-woo-elta-courier' );
				return $result;
			}

			require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
		
			if ($this->iben_elta_main_settings_enable_test_mode){
				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR .  'src/elta_wsdl_test/CREATEAWB.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => Iben_createClassMap::get(),
				);
			}else{
				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR .  'src/elta_wsdl/CREATEAWB.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => Iben_createClassMap::get(),
				);	
			}
			
			
				
			$rEAD = new \CreateService\Iben_createREAD($options);				

			$set = new \CreateStruct\Iben_createREAD;	


			// 1 = default 
			if ( $types['second_account'] == 1 ){	

				$customer = $this->iben_elta_main_settings_customer;
				$username = $this->iben_elta_main_settings_username;
				$password = $this->iben_elta_main_settings_password;
				$sub_password = $this->iben_elta_main_settings_sub_password;
			}else{

				$iben_elta_main_settings_fields = $this->main_settings;

				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_customer'] ) ) {
					$customer = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_customer'] );
				}else{
					$customer = '';
				}
				
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_username'] ) ) {
					$username = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_username'] );
				}else{
					$username = '';
				}
				
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_password'] ) ) {
					$password = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_password'] );
				}else{
					$password = '';
				}
		
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_sub_password'] ) ) {
					$sub_password = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_sub_password'] );
				}else{
					$sub_password = '';
				}

			}


			if ( $types['subcode'] != '' ){
				$sub_password = $types['subcode'];
			}
			
			$set->setPel_apost_code( $customer );
			$set->setPel_user_code( $username );
			$set->setPel_user_pass( $password );	
			$set->setPel_apost_sub_code( $sub_password );

			$set->setPel_paral_name( $types['fullname'] );
			$set->setPel_paral_address( $types['address'] );
			$set->setPel_paral_area( $types['region'] );
			$set->setPel_paral_tk( $types['postal_code'] );
			$set->setPel_paral_thl_1( $types['phone1'] );
			$set->setPel_paral_thl_2( $types['phone2'] );

			$set->setPel_service( $types['service'] );

			$set->setPel_ant_poso( $types['cash_delivery']);

			$set->setPel_baros( $types['weight'] );
			$set->setPel_temaxia( $types['packages_number'] );
			$set->setPel_paral_sxolia( $types['comments']  );

			//ΕΙΔΙΚΗ ΜΕΤΑΧΕΙΡΙΣΗ
			$set->setPel_sur_1( isset($types['sur_1']) ? $types['sur_1'] : '0' );

			// ΕΠΙΔΟΣΗ ΠΡΟΚΑΘΡΙΣΜΕΝΗΣ ΩΡΑΣ
			$set->setPel_sur_2( $types['sur_2'] );

			// ΕΠΙΔΟΣΗ ΣΑΒΒΑΤΟΥ
			$set->setPel_sur_3( $types['sur_3'] );
			
			$set->setPel_ref_no( $types['ref'] );
			
			
			$action = $rEAD->READ($set);

			if ( $action !== false ){
				if ( $rEAD->getResult() ){
				
					if ( $action->getSt_flag() == 0 ){

						$result['state'] = true;
						$result['result'] = $action;
						return $result;

					}else if ( $action->getSt_flag() > 0 ){
						$result['state'] = false;
						$result['result'] = $action->getSt_title();
						return $result;
					}
				}else{
					$error = '';
					$result['state'] = false;
					
					if ($rEAD->getLastError()){
						
						foreach ($rEAD->getLastError() as $error){
							$error.= $error.'<br/>';
						}
					}
					$result['result'] = '1. OTHER ERROR CONNECTION. Try in a minute again';
					$result['error'] = $error;
					return $result;				
				}

			}else{
				$error = '';
				$error_ = array();
				$result['state'] = false;
				
				if ($rEAD->getLastError()){
					
					foreach ($rEAD->getLastError() as $error){
						$error_[] = $error->faultstring;
					}

					$result['result'] = $error_['0'].' Try again a little later';
					$result['error'] = $error;

					return $result;

				}else{

					$result['state'] = false;
					$result['result'] = 'UKNOWN ERROR';
					return $result;						

				}

			}

			
		}


		/**
		 *
		 * @since     1.0.0
		 * 
		 */
		private function ibenEltaWebServiceGenerateVoucherPDF($voucher, $id_account = 1){
			
			if ( $this->iben_elta_check_logins() ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Your ELTA Courier logins required', 'iben-woo-elta-courier' );
				return $result;
			}
			if ( !extension_loaded('soap') ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Soap PHP extension must be enabled', 'iben-woo-elta-courier' );
				return $result;
			}

			require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
			if ($this->iben_elta_main_settings_enable_test_mode){
				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR .  'src/elta_wsdl_test/PELB64VG.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => \Iben_printClassMap::get(),
				);
			}else{
				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR .  'src/elta_wsdl/PELB64VG.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => \Iben_printClassMap::get(),
				);				
			}

			$rEAD = new \PrintService\Iben_printREAD($options);
			$set = new \PrintStruct\Iben_printREAD;
			
			// 1 = default 
			if ( $id_account == 1 ){	

				$customer = $this->iben_elta_main_settings_customer;
				$username = $this->iben_elta_main_settings_username;
				$password = $this->iben_elta_main_settings_password;
				$sub_password = $this->iben_elta_main_settings_sub_password;
			}else{

				$iben_elta_main_settings_fields = $this->main_settings;

				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_customer'] ) ) {
					$customer = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_customer'] );
				}else{
					$customer = '';
				}
				
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_username'] ) ) {
					$username = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_username'] );
				}else{
					$username = '';
				}
				
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_password'] ) ) {
					$password = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_password'] );
				}else{
					$password = '';
				}
		
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_sub_password'] ) ) {
					$sub_password = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_sub_password'] );
				}else{
					$sub_password = '';
				}

			}

			$set->setPel_apost_code( $customer );
			$set->setPel_user_code( $username );
			$set->setPel_user_pass( $password );

			
			$set->setVg_code( $voucher );
			$set->setPaper_size( $this->iben_elta_main_settings_papersize );
			
			$news = $rEAD->READ($set);

			if ($news !== false ){
				if ( $rEAD->getResult() ){			
					if ( $news->getSt_flag() == 0 ){
						//print_r($rEAD->getResult());
						$result = Array();
						$base_64 = $news->getB64_string();

						// we give the file VOUCHER name
						$name    = $voucher.".pdf";
					
						// a route is created, (it must already be created in its repository(pdf)).
						$route = IBEN_WOO_ELTA_UPLOAD_FOLDER.$name;
					
						// decode base64
						$pdf_b64 = base64_decode($base_64);

							// you record the file in existing folder
							if(file_put_contents($route, $pdf_b64)){

								$result['state'] = true;
								$result['voucher'] = $voucher;
								$result['result'] = $route;
								return $result;

							}else{
								$result['state'] = false;
								$result['result'] = esc_html__('Was unable to put content in the PDF. Do you have enough space in your server?', 'iben-woo-elta-courier');
								return $result;
							}

					}else if ( $news->getSt_flag() > 0 ){

						$result['state'] = false;

						$result_error = $news->getSt_title();
		
						if (strpos($result_error, '0901') !== false) {
							$result['result'] = esc_html__('You cannot regenerate a PDF that had been generated with different size.', 'iben-woo-elta-courier');
						}else{
							$result['result'] = $news->getSt_title();
						}

						return $result;
					}
				}else{
					$error = '';
					$error_ = array();
					$result['state'] = false;
					
					if ($rEAD->getLastError()){
						
						foreach ($rEAD->getLastError() as $error){
							$error_[] = $error->faultstring;
						}

						$result['result'] = $error_['0'].' Try again a little later';
						$result['error'] = $error;
						return $result;

					}else{

						$result['state'] = false;
						$result['result'] = 'UKNOWN ERROR';
						return $result;
					}

					
				}

			} else {

				$error = '';
				$error_ = array();
				$result['state'] = false;
				
				if ($rEAD->getLastError()){
					
					foreach ($rEAD->getLastError() as $error){
						$error_[] = $error->faultstring;
					}
					$result['result'] = $error_['0'].' Try again a little later';
					
					$result['error'] = $error;
					return $result;

				}else{

					$result['state'] = false;
					$result['result'] = 'UKNOWN ERROR';
					return $result;
				}
				

			}	
		}


		/**
		 *
		 * @since     1.0.0
		 * 
		 */
		private function ibenEltaWebServiceTracing($voucher, $ref='1', $id_account = 1){

			if ( $this->iben_elta_check_logins() ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Your ELTA Courier logins required', 'iben-woo-elta-courier' );
				return $result;
			}
			if ( !extension_loaded('soap') ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Soap PHP extension must be enabled', 'iben-woo-elta-courier' );
				return $result;
			}
			
			require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';

			if ($this->iben_elta_main_settings_enable_test_mode){
				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR . 'src/elta_wsdl_test/PELTT01.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => \Iben_tracingClassMap::get(),
				);
			}else{

				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR . 'src/elta_wsdl/PELTT01.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => \Iben_tracingClassMap::get(),
				);

			}


			$rEAD = new \TracingService\Iben_tracingREAD($options);
			$set = new \TracingStruct\Iben_tracingREAD;
			



			if ( $id_account == 1 ){	

				$customer = $this->iben_elta_main_settings_customer;
				$username = $this->iben_elta_main_settings_username;
				$password = $this->iben_elta_main_settings_password;
				$sub_password = $this->iben_elta_main_settings_sub_password;
			}else{

				$iben_elta_main_settings_fields = $this->main_settings;

				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_customer'] ) ) {
					$customer = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_customer'] );
				}else{
					$customer = '';
				}
				
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_username'] ) ) {
					$username = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_username'] );
				}else{
					$username = '';
				}
				
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_password'] ) ) {
					$password = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_password'] );
				}else{
					$password = '';
				}
		
				if( isset( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_sub_password'] ) ) {
					$sub_password = sanitize_text_field( $iben_elta_main_settings_fields['iben_elta_main_settings_2nd_sub_password'] );
				}else{
					$sub_password = '';
				}
			}
			
			$set->setWpel_code( $customer );
			$set->setWpel_user( $username );
			$set->setWpel_pass( $password );

			if ($ref == 1){ // voucher
				$set->setWpel_vg( $voucher );
			}else if ($ref == 2){ //ref
				$set->setWpel_ref( $voucher );
			}
			
			$set->setWpel_flag( $ref );

			$action = $rEAD->READ($set);

			if ($action !== false ){

				if ($action->getSt_flag() == 0 ){
					$result['state'] = true;
					$result['result'] = $action;
					return $result;
				}else{
					$result['state'] = false;
					$result['result'] =  $action->getSt_title();
					return $result;
				}
			}else{

				$error = '';
				$error_ = array();
				$result['state'] = false;
				
				if ($rEAD->getLastError()){
					
					foreach ($rEAD->getLastError() as $error){
						$error_[] = $error->faultstring;
					}
					
					$result['result'] = $error_['0'].' Try again a little later';
					$result['error'] = $error;
					return $result;					
					
				}else{

					$result['state'] = false;
					$result['result'] = 'UKNOWN ERROR';
					return $result;
				}


			}	
			
		}

		/**
		 *
		 * @since     1.7.4
		 * 
		 */		
		function iben_elta_frontend_myaccount_orders_actions( $actions, $order ) {

			$prefix = sanitize_text_field($this->main_settings['iben_elta_main_settings_frontend_tracking_action_link_text']);
			$prefix = (isset($prefix) ? $prefix: '');
			$order_id  = $order->get_id();

			$get_vouchers = iben_elta_get_Vouchers_from_id_order($order_id);
			if (!empty($get_vouchers) && is_array($get_vouchers) ){
				foreach ($get_vouchers as $index => $voucher){
					$vouch = json_decode($voucher->elta_delivery_vouchers);
					$vouch = $vouch['0'];
		
					$url = "https://www.elta-courier.gr/search?br=$vouch";
		
					$actions['iben_elta_voucher_my_account_orders_action'.$index ] = array(
						'url'  => $url,
						'name' => $prefix.' '.$vouch
					);
				}
			}
		
			return $actions;
		}

		/**
		 *
		 * @since     1.7.4
		 * 
		 */			
		function iben_elta_frontend_myaccount_orders_js() {
			$action_slug = 'iben_elta_voucher_my_account_orders_action';
			?>
			<script>
			jQuery(function($){
				$('a.<?php echo $action_slug; ?>0').each( function(){
					$(this).attr('target','_blank');
					$(this).addClass('iben_elta_voucher_my_account_orders_action');
				})
				$('a.<?php echo $action_slug; ?>1').each( function(){
					$(this).attr('target','_blank');
					$(this).addClass('iben_elta_voucher_my_account_orders_action');
				})		
			});
			</script>
			<?php
		}

		/**
		 *
		 * @since     1.7.4
		 * 
		 */	
		function iben_elta_frontend_myaccount_addtracing( $order_id ){

			$get_vouchers = iben_elta_get_Vouchers_from_id_order($order_id);

			if (!empty($get_vouchers) && is_array($get_vouchers) ){

				$title = sanitize_text_field($this->main_settings['iben_elta_main_settings_frontend_view_order_tracking_tracing_title']);
				$title = (isset($title ) ? $title : '');

				echo '<div class="iben_elta_my_accounts_view_order">';
					if ($title){
						echo '<h2 class="iben_elta_my_accounts_view_order_voucher_tracing_title">'.$title.'</h2>';
					}
					$vouchersexport = '';
					foreach ($get_vouchers as $index => $voucher){

						$vouch = json_decode($voucher->elta_delivery_vouchers);
						$vouch = $vouch['0'];

						echo '<div class="iben_elta_my_accounts_view_order_voucher_tracing_block">';
						echo '<span class="iben_elta_my_accounts_view_order_voucher_tracing_block_title">'.$vouch.'</span>';

						$news = $this->ibenEltaWebServiceTracing( $vouch, '1', $voucher->id_elta_account);
						if ($news['state'] === true){
								
							$news = $news['result'];
							$route = '';
							$array_results = $news->getWeb_status();
							if (is_array($array_results) && count($array_results) > 0){
								foreach ($array_results as $item){

									if( !$item->getWeb_status_title() ) {
										continue;
									}
						
									$route.= '<span style="display: block;" class="iben_elta_my_accounts_view_order_voucher_tracing_block_element">'.$item->getWeb_status_title().'</span>';
									$route.= '<span class="iben_elta_my_accounts_view_order_voucher_tracing_block_date">'.substr($item->getWeb_date(), 6, 2).'-'.substr($item->getWeb_date(), 4, 2).'-'.substr($item->getWeb_date(), 0, 4).' '.substr($item->getWeb_time(), 0, 2).':'.substr($item->getWeb_time(), 2, 2).'</span>';
								}
							}

							echo $route;

						}
						echo '</div>';
					}
				echo '</div>';
			}
		}


		/**
		 * Check cron deliverity
		 *
		 * @since     1.2.0
		 */
		function iben_elta_cron_check_delivery_hook(){
			
			$limit = 10;
			$vouchers = '';

            global $wpdb;
			$table_name = $wpdb->prefix . 'iben_elta_vouchers';
			
			$main_settings = $this->main_settings;
			$limit = sanitize_text_field($main_settings['iben_elta_main_settings_cron_query_limit']);

			if ( isset($limit) ){
				$limit = $limit;
			}
			$limit = apply_filters('iben_elta_filter_cron_query_limit', $limit);
			

			$all_undelivered = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE delivered != 1 AND deleted = 0 ORDER BY last_check_delivered_time ASC LIMIT %d", $limit));

			if ($all_undelivered){

				foreach ($all_undelivered as $record){

					$index_delivery = 0;

					if ($record->id_elta_reason == 1){
						$results = json_decode($record->elta_delivery_vouchers);
						foreach ($results as $voucher){
							// get only 1 voucher in case are more that 1 parcel
							if ($index_delivery == 0){

								$check_tracing = $this->ibenEltaWebServiceTracing( $voucher, '1', $record->id_elta_account);

								if ($check_tracing['state'] === true){

									$check_tracing = $check_tracing['result'];
									$alternative_delievered = false;
									$risk_returning = false;
									$returned = false;
									$delivered_name = $check_tracing->getPod_name();
					
									$getWeb_status = $check_tracing->getWeb_status();
									$getWeb_status = $getWeb_status['0'];
									$getWeb_status = $getWeb_status->getWeb_status_title();
						
					
									$array = array('ΑΠΟΣΤΟΛΗ ΠΑΡΑΔΟΘΗΚΕ-ΧΩΡΙΣ ΣΤΟΙΧΕΙΟ ΠΑΡΑΔΟΣΗΣ');
									if (in_array($getWeb_status, apply_filters('iben_elta_filter_delivery_array_strings', $array), true) === true){
										$alternative_delievered = true;
									}
									
									$array_returning = array('ΕΙΣΑΓΩΓΗ ΣΤΗ ΘΥΡΙΔΑ', 'ΕΙΣΑΓΩΓΗ ΘΥΡΙΔΑ', 'ΠΑΡΑΛΗΠΤΗΣ ΕΚΤΟΣ - ΕΤΑΙΡΙΑ ΚΛΕΙΣΤΗ',  'Ο ΠΑΡΑΛΗΠΤΗΣ ΔΕΝ ΕΝΤΟΠΙΖΕΤΑΙ', 'ΠΑΡΑΛΗΠΤΗΣ ΔΕΝ ΕΝΤΟΠΙΖΕΤΑΙ', 'ΕΙΔΟΠΟΙΗΘΗΚΕ - ΕΙΣΑΓΩΓΗ ΣΤΟΝ ΣΤΑΘΜΟ');
									if (in_array($getWeb_status, apply_filters('iben_elta_filter_returning_array_strings', $array_returning), true) === true){
										$risk_returning = true;
									}
									
									$array_returned = array('ΕΠΙΣΤΡΟΦΗ ΣΤΟΝ ΑΠΟΣΤΟΛΕΑ', 'ΑΚΥΡΩΣΗ ΑΠΟ ΔΙΑΝΟΜΕΑ');
									if (in_array($getWeb_status, apply_filters('iben_elta_filter_returned_array_strings', $array_returned), true) === true){
										$returned = true;
									}

									if ($delivered_name || $alternative_delievered){
										
										$voucher_info = maybe_unserialize($record->elta_voucher_info);
										$fullname = $voucher_info['fullname'];

										$date = substr($check_tracing->getPod_date(), 6, 2).'-'.substr($check_tracing->getPod_date(), 4, 2).'-'.substr($check_tracing->getPod_date(), 0, 4);
										$time = substr($check_tracing->getPod_time(), 0, 2).':'.substr($check_tracing->getPod_time(), 2, 2);

										$date_created = date_i18n(get_option('date_format'), $record->time) .' | '. date_i18n(get_option('time_format'), $record->time);

										$vouchers .= '<p>'.esc_html__( 'Date created', 'iben-woo-elta-courier' ).': '.$date_created.'<br/>'.esc_html__( 'SYDETA', 'iben-woo-elta-courier' ).': '.$voucher.'<br/>'.esc_html__( 'Full name', 'iben-woo-elta-courier' ).': '.$fullname.'<br/>'.esc_html__( 'Delivered date', 'iben-woo-elta-courier' ).': '.$date.' '.$time.'<br/></p>';

										$date = substr($check_tracing->getPod_date(), 6, 2).'-'.substr($check_tracing->getPod_date(), 4, 2).'-'.substr($check_tracing->getPod_date(), 0, 4);
										$time = substr($check_tracing->getPod_time(), 0, 2).':'.substr($check_tracing->getPod_time(), 2, 2);
										
														
										$updatedata = [ 'delivered_name' => $delivered_name, 'delivered_time' => $time, 'delivered_date' => $date, 'delivered' => '1',  'last_check_delivered_time' => current_time('timestamp')]; 						 
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );
										
										do_action( 'iben_elta_check_delivery_hook_after_delivered', $record->id_order, $record->id);
										
									}elseif ( $risk_returning){

										$updatedata = [ 'delivered' => '3',  'last_check_delivered_time' => current_time('timestamp')]; 						 
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );
					
										do_action( 'iben_elta_check_delivery_hook_after_riskreturning', $record->id_order, $record->id);
										
									}elseif ( $returned){

										$updatedata = [ 'delivered' => '2',  'last_check_delivered_time' => current_time('timestamp')]; 						 
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );
					
										do_action( 'iben_elta_check_delivery_hook_after_returned', $record->id_order, $record->id);
										
									}else{

										// NOT DELIVERED				
										$updatedata = [ 'delivered' => '-1', 'last_check_delivered_time' => current_time('timestamp')]; 						 
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );
										
									}					
								}else{

									$alternative_canceled = false;
									if ( strpos($check_tracing['result'], 'Λάθος SYDETA') !== false ) {
										$alternative_canceled = true;
									}
				
									if ($alternative_canceled){
										global $wpdb;
										$table_name = $wpdb->prefix . 'iben_elta_vouchers';
										$updatedata = [ 'deleted' => '1', 'last_check_delivered_time' => current_time('timestamp')];					
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );		
									}

								}

							}
							$index_delivery++;
						} 
					}


					$index_returned = 0;

					if ($record->id_elta_reason == 2){
						$results = json_decode($record->elta_returned_vouchers);

						foreach ($results as $voucher){
							// get only 1 voucher in case are more that 1 parcel
							if ($index_returned == 1){

								$check_tracing = $this->ibenEltaWebServiceTracing( $voucher, '1', $record->id_elta_account);

								if ($check_tracing['state'] === true){

									$check_tracing = $check_tracing['result'];
									$alternative_delievered = false;
									$delivered_name = $check_tracing->getPod_name();
					
									$getWeb_status = $check_tracing->getWeb_status();
									$getWeb_status = $getWeb_status['0'];
									$getWeb_status = $getWeb_status->getWeb_status_title();
						
					
									$array = array('ΑΠΟΣΤΟΛΗ ΠΑΡΑΔΟΘΗΚΕ-ΧΩΡΙΣ ΣΤΟΙΧΕΙΟ ΠΑΡΑΔΟΣΗΣ');
					
									if (in_array($getWeb_status, apply_filters('iben_elta_filter_delivery_array_strings', $array), true) === true){
										$alternative_delievered = true;
									}
					
									
									
									if ($delivered_name || $alternative_delievered){
										
										$voucher_info = maybe_unserialize($record->elta_voucher_info);
										$fullname = $voucher_info['fullname'];

										$date = substr($check_tracing->getPod_date(), 6, 2).'-'.substr($check_tracing->getPod_date(), 4, 2).'-'.substr($check_tracing->getPod_date(), 0, 4);
										$time = substr($check_tracing->getPod_time(), 0, 2).':'.substr($check_tracing->getPod_time(), 2, 2);

										$date_created = date_i18n(get_option('date_format'), $record->time) .' | '. date_i18n(get_option('time_format'), $record->time);

										$vouchers .= '<p>'.esc_html__( 'Date created', 'iben-woo-elta-courier' ).': '.$date_created.'<br/>'.esc_html__( 'SYDETA', 'iben-woo-elta-courier' ).': '.$voucher.'<br/>'.esc_html__( 'Full name', 'iben-woo-elta-courier' ).': '.$fullname.'<br/>'.esc_html__( 'Delivered date', 'iben-woo-elta-courier' ).': '.$date.' '.$time.'<br/></p>';
										
										

										$date = substr($check_tracing->getPod_date(), 6, 2).'-'.substr($check_tracing->getPod_date(), 4, 2).'-'.substr($check_tracing->getPod_date(), 0, 4);
										$time = substr($check_tracing->getPod_time(), 0, 2).':'.substr($check_tracing->getPod_time(), 2, 2);
										
														
										$updatedata = [ 'delivered_name' => $delivered_name, 'delivered_time' => $time, 'delivered_date' => $date, 'delivered' => '1',  'last_check_delivered_time' => current_time('timestamp')]; 						 
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );
										
										do_action( 'iben_elta_check_delivery_hook_after_delivered', $record->id_order, $record->id);

									}else{
										
										// NOT DELIVERED				
										$updatedata = [ 'delivered' => '-1', 'last_check_delivered_time' => current_time('timestamp')]; 						 
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );
										
									}
					
								}else{
									
									$alternative_canceled = false;
									if ( strpos($check_tracing['result'], 'Λάθος SYDETA') !== false ) {
										$alternative_canceled = true;
									}
				
									if ($alternative_canceled){
										global $wpdb;
										$table_name = $wpdb->prefix . 'iben_elta_vouchers';
										$updatedata = [ 'deleted' => '1', 'last_check_delivered_time' => current_time('timestamp')];					
										$where = [ 'id' => absint($record->id) ];
										$db_update = $wpdb->update( $table_name, $updatedata, $where );		
									}									

								}

							}
							$index_returned++;
						} 
					}



				}
			}
			
			$admin_email = sanitize_text_field($main_settings['iben_elta_main_settings_admin_email_cron_notification']);		

			if ( isset($main_settings['iben_elta_main_settings_enable_email_cron_notification']) && isset($admin_email) ){

				if ($vouchers){

					$to = $admin_email;

					$subject = 'Cron run - ELTA vouchers delivered';
					$headers = array('Content-Type: text/html; charset=UTF-8');
					
					$body = '<p>Δέματα που παραδόθηκαν. </p>'.$vouchers;
					
					wp_mail( $to, $subject, $body, $headers );
				}
			}

			return true;
		}
		
		/**
		 * Check cron delete PDFs
		 *
		 * @since     1.4.0
		 */
		function iben_elta_cron_delete_pdfs_hook(){

			// $main_settings = get_option('iben_elta_main_settings_fields'); 
			$main_settings = $this->main_settings;
			$i = sanitize_text_field($main_settings['iben_elta_main_settings_delete_pdf_cron_query']);
			
			$limit = 50;
			
			$limit = apply_filters('iben_elta_filter_cron_delete_pdf_limit', $limit);

			global $wpdb;
			$table_name = $wpdb->prefix . 'iben_elta_vouchers';

			switch ($i) {
				case 1:
					$queries = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE delivered > 0 AND pdf_deleted = 0 LIMIT %d", $limit));
					break;
				case 2:
					$time =  strtotime("-1 week");
					$queries = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE delivered > 0 AND pdf_deleted = 0 AND time <= %d LIMIT %d", $time, $limit));
					break;
				case 3:
					$time =  strtotime("-2 weeks");
					$queries = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE delivered > 0 AND pdf_deleted = 0 AND time <= %d LIMIT %d", $time, $limit));
					break;
				case 4:
					$time =  strtotime("-1 month");
					$queries = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE delivered > 0 AND pdf_deleted = 0 AND time <= %d LIMIT %d", $time, $limit));
					break;

				default:
					$time =  strtotime("-1 week");
					$queries = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE delivered > 0 AND pdf_deleted = 0 AND time <= %d LIMIT %d", $time, $limit));

			}


			$countdeleted = 0;
			if ($queries){

				foreach ($queries as $record){

					
					if ($record->id_elta_reason == 1){

						$results = json_decode($record->elta_delivery_vouchers);
						foreach ($results as $voucher){
							
							$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$voucher.".pdf";
							
							if(is_file($file)){
								if ( iben_elta_secure_unlink($file) ) {
									$countdeleted++;
								}
							}
							
						}

						// check if there is compined PDF
						$compined = $record->elta_delivery_compined_vouchers;
						if ($compined != ''){

							$compined_c = IBEN_WOO_ELTA_UPLOAD_FOLDER.$compined.".pdf";
							
							if(is_file($compined_c)){
								if ( iben_elta_secure_unlink($compined_c) ) {
									$countdeleted++;
								}
							}

						}

					}


					if ($record->id_elta_reason == 2){

						$results = json_decode($record->elta_returned_vouchers);

						foreach ($results as $voucher){

							$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$voucher.".pdf";
							
							if(is_file($file)){
								if ( iben_elta_secure_unlink($file) ) {
									$countdeleted++;
								}
							}

						}

					
						// check if there is compined PDF
						$compined = $record->elta_returned_compined_vouchers;
						if ($compined != '' ){

							$compined_c = IBEN_WOO_ELTA_UPLOAD_FOLDER.$compined.".pdf";
							
							if(is_file($compined_c)){
								if ( iben_elta_secure_unlink($compined_c) ) {
									$countdeleted++;
								}
							}

						}						


					}

					//UPDATE RECORD. SET pdf_deleted = 1 so next cron will skip this record from query
					$updatedata = [ 'pdf_deleted' => '1'];
					$where = [ 'id' => $record->id ];
					$wpdb->update( $table_name, $updatedata, $where );


				}//end queries
			}


		}




		/**
		 *
		 * @since     1.0.0
		 * 
		 */
		private function ibenEltaWebServiceStation($postal_code){

			if ( $this->iben_elta_check_logins() ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Your ELTA Courier logins required', 'iben-woo-elta-courier' );
				return $result;
			}
			if ( !extension_loaded('soap') ){
				$result['state'] = false;
				$result['result'] =  esc_html__( 'Soap PHP extension must be enabled', 'iben-woo-elta-courier' );
				return $result;
			}
			
			require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
			
			if ($this->iben_elta_main_settings_enable_test_mode){
				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR . 'src/elta_wsdl_test/PELSTATION.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => \Iben_stationClassMap::get(),
				);
			}else{
				$options = array(
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_URL => IBEN_WOO_ELTA_ROOT_DIR . 'src/elta_wsdl/PELSTATION.wsdl',
					\WsdlToPhp\PackageBase\AbstractSoapClientBase::WSDL_CLASSMAP => \Iben_stationClassMap::get(),
				);				
			}

			$rEAD = new \StationService\Iben_stationREAD($options);

			$set = new \StationStruct\Iben_stationREAD;
			$set->setWeb_tk( $postal_code );
			
			
			$action = $rEAD->READ($set);

			if ($action !== false ){

				if ($action->getSt_flag() == 0 ){

					$result['state'] = true;
					$result['result'] = $action;
					return $result;
				}else{

					$result['state'] = false;
					$result['result'] =  $action->getSt_title();
					return $result;

				}
				
			}else{

				$error = '';
				$error_ = array();
				$result['state'] = false;
				
				if ($rEAD->getLastError()){
					
					foreach ($rEAD->getLastError() as $error){
						// $error.= $error.'<br/>';
						$error_[] = $error->faultstring;
					}
					$result['result'] = $error_['0'].' Try little later again';
					$result['error'] = $error;
					return $result;

				}else{

					$result['state'] = false;
					$result['result'] = 'UKNOWN ERROR';
					return $result;
				}


			}

		}	

		public function iben_elta_settings_remove_bulk_ajax_handler(){
		
			if (isset($_POST['iben_data'])) {
				
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				
				$query = iben_elta_get_BulktoDelete();

				$export = '';


				global $wpdb;
				$table_name = $wpdb->prefix . 'iben_elta_bulk_vouchers';
				$table_name2 = $wpdb->prefix . 'iben_elta_vouchers';

				$countdeleted = 0;

				foreach ($query as $qry){

					if ($qry->compined_pdf != ''){

						$file2 = IBEN_WOO_ELTA_UPLOAD_FOLDER.$qry->compined_pdf.".pdf";
							
							if(is_file($file2)){
								if ( iben_elta_secure_unlink($file2) ) {
									$countdeleted++;
								}
							}

							$wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE id = %d", $qry->id));
						
						}

				}


				$limit = 20;
				$limit = apply_filters('iben_elta_filter_cron_bulk_delete_pdf_limit', $limit);


				$time =  strtotime("-1 month");
				$queries = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name2 WHERE delivered > 0 AND pdf_deleted = 0 AND time <= %d LIMIT %d", $time, $limit));

				if ($queries){

					foreach ($queries as $record){
	
						if ($record->id_elta_reason == 1){
	
							$results = json_decode($record->elta_delivery_vouchers);
							foreach ($results as $voucher){
								
								$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$voucher.".pdf";
								
								if(is_file($file)){
									if ( iben_elta_secure_unlink($file) ) {
										$countdeleted++;
									}
								}
								
							}
	
							// check if there is compined PDF
							$compined = $record->elta_delivery_compined_vouchers;
							if ($compined != ''){
	
								$compined_c = IBEN_WOO_ELTA_UPLOAD_FOLDER.$compined.".pdf";
								
								if(is_file($compined_c)){
									if ( iben_elta_secure_unlink($compined_c) ) {
										$countdeleted++;
									}
								}
	
							}
	
						}
	
						if ($record->id_elta_reason == 2){
	
							$results = json_decode($record->elta_returned_vouchers);
	
							foreach ($results as $voucher){
	
								$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$voucher.".pdf";
								
								if(is_file($file)){
									if ( iben_elta_secure_unlink($file) ) {
										$countdeleted++;
									}
								}
	
							}
	
							// check if there is compined PDF
							$compined = $record->elta_returned_compined_vouchers;
							if ($compined != '' ){
	
								$compined_c = IBEN_WOO_ELTA_UPLOAD_FOLDER.$compined.".pdf";
								
								if(is_file($compined_c)){
									if ( iben_elta_secure_unlink($compined_c) ) {
										$countdeleted++;
									}
								}
	
							}						
	
	
						}
	
						//UPDATE RECORD. SET pdf_deleted = 1 so next cron will skip this record from query
						$updatedata = [ 'pdf_deleted' => '1'];
						$where = [ 'id' => $record->id ];
						$wpdb->update( $table_name2, $updatedata, $where );
				

					}//end foreach queries

				}


				$result1 = $countdeleted.' '.esc_html__( 'PDF Deleted', 'iben-woo-elta-courier' );

				$resultss = array (
					"status" => "success",
					"result" => $result1
				);
		
				echo json_encode ( $resultss );
		
			}
		
			// ajax handlers must die
			wp_die();
		}	


		/**
		 * Attach tracking info on completed email when hooked
		 *
		 * @since     1.1.0
		 * 
		 */
		public function iben_elta_email_send_tracking_info_handler( $order, $sent_to_admin, $plain_text, $email ) {
			
			if ( $email->id != apply_filters( 'iben_elta_email_send_tracking_emailid_filter', 'customer_completed_order', $order ) ) {
				return;
			}
			
	
			if (!is_object($order)){
				return;
			}
			
			$order_id  = $order->get_id();			
			
			$order_notification_completed = $order->get_meta( 'iben_elta_tracking_order_notification', true); 				
			
			$order_notification_completed = apply_filters('iben_elta_email_send_tracking_filter', $order_notification_completed, $order_id);

			if ( !$order_notification_completed ){
				return;
			}

			global $wpdb;
			$table_name = $wpdb->prefix . 'iben_elta_vouchers';

			$all_records = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name WHERE id_order = %d AND id_elta_reason = 1 AND deleted = 0", $order_id));  

			
			$vouchers = '';
			
			if ($all_records){

				foreach ($all_records as $record){

					$results = json_decode($record->elta_delivery_vouchers);
					$index = 0;
					foreach ($results as $voucher){
						// get only 1 voucher in case are more that 1 parcel
						if ($index == 0){
							$vouchers .= '<a href="https://www.elta-courier.gr/search?br='.$voucher.'" target="_blank">'.$voucher.'</a> ';
						}
						$index++;
					} 

					//update email_sent to 1
					$updatedata = [ 'email_sent' => '1']; 						 
					$where = [ 'id' => $record->id ];
					$db_update = $wpdb->update( $table_name, $updatedata, $where );	
					
				}

			}

			$message = '';
			$options = get_option( 'iben_elta_main_settings_fields' );
			
			if ( isset($options['iben_elta_main_settings_email_default_body']) && $options['iben_elta_main_settings_email_default_body'] != '' ){
				$message = $options['iben_elta_main_settings_email_default_body'];
			}

			if ($vouchers){
			
				$search = array("[iben_elta_tracking]");
				$replacewith = array( $vouchers);

				$replaced_message = apply_filters( 'iben_elta_send_traching_message_before_filter', $message, $order );

				$message = str_replace($search, $replacewith, $replaced_message);

				$allowed_html = array(
					'a' => array(
						'href' => array(),
						'title' => array(),
						'alt' => array(),
					),
					'br' => array(),
					'em' => array(),
					'strong' => array(),
					'ul' => array(),
					'li' => array(),
					'ol' => array(),
					'p' => array(),
					'h1' => array(),
					'h2' => array(),
					'h3' => array(),
					'h4' => array(),
					'h5' => array(),
					'h6' => array(),
					'i' => array(),
				);

				$allowed_html = apply_filters( 'iben_elta_email_tracking_metabox_email_allowed_html_filter', $allowed_html, $order );
				
				$message = wp_kses($message, $allowed_html);

				echo $message;
			}

	
		}		


		/**
		 * Display at admin edit order checkbox for sending tracking info
		 *
		 * @since     1.1.0
		 */
		function iben_elta_display_checkbox_send_tracking_info( $order ){
		
			$woo_settings = $this->main_settings;
			$is_default = isset( $woo_settings['iben_elta_main_settings_email_checked_default_completed'] ) ? (int)$woo_settings['iben_elta_main_settings_email_checked_default_completed'] : 0;	
			
			$checked = '';

			if ( $order->get_meta( 'iben_elta_tracking_order_notification', true) == 1 ){
				$checked = 'checked="checked"';
			}else{
				
				if ( $is_default && !$order->meta_exists('iben_elta_tracking_order_notification') ){
					$checked = 'checked="checked"';
				}
				
			}

			?>

			<div class="form-field form-field-wide ">
	
				<h3><?php esc_html_e( 'Include ELTA tracking when order completed', 'iben-woo-elta-courier' ); ?></h3>

				<div class="address">  
		
					<label for="iben_elta_tracking_order_notification" style="display: contents;">
						<b><?php esc_html_e( 'Checked, will attached to completed email', 'iben-woo-elta-courier' ); ?>:</b>
					</label>
						<input
							id="iben_elta_tracking_order_notification"
							name="iben_elta_tracking_order_notification"
							type="checkbox"
							value="1"
							style="width: 20px;"
							<?php echo $checked; ?> />

					</div>			
	
				</div>
			
			<?php			
		}

		
		/**
		 * Save checkbox on edit order
		 *
		 * @since     1.1.0
		 */
		public function iben_elta_save_checkbox_send_tracking_info( $order_id ){
			if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
				return;
			}
	
			if (defined('DOING_AJAX') && DOING_AJAX) {
				return;
			}
			$order = wc_get_order( $order_id );

			if ( !is_a( $order, 'WC_Order' ) ) return;

			if ( isset( $_POST['iben_elta_tracking_order_notification'] ) ) {
				$order->update_meta_data( 'iben_elta_tracking_order_notification', 1);	
			}else{
				$order->update_meta_data( 'iben_elta_tracking_order_notification', 0);	
			}
			$order->save();	
			return true;

		}

		
		/**
		 * Save checkbox with AJAX on edit order
		 *
		 * @since     1.1.3
		 */
		public function iben_elta_change_postmeta_attach_tracking_info_email_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		

				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				
				$order = wc_get_order( absint( $data['order_id'] ) );

				if ( !is_a( $order, 'WC_Order' ) ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Order does not exist', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();	
				}	
				
				$order_id  = $order->get_id();

				$update_email_completed = absint($data['update_email_completed']);
				$message = '';
				if( $update_email_completed == 1 || $update_email_completed == 0){

					$order->update_meta_data( 'iben_elta_tracking_order_notification', $update_email_completed);	
					$order->save();	

					$message = esc_html__('Updated', 'iben-woo-elta-courier');

				}

				

				$resultss = array (
					"status" => "success",
					"result" => $message,
					"update_email_completed" => $update_email_completed
				);
		
				echo json_encode ( $resultss );
					
			}
		
			// ajax handlers must die
			wp_die();
		}

		/**
		 * Save SMS checkbox with AJAX on edit order
		 *
		 * @since     1.5
		 */
		public function iben_elta_change_postmeta_send_sms_completed_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				
				$order = wc_get_order( absint( $data['order_id'] ) );

				if ( !is_a( $order, 'WC_Order' ) ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Order does not exist', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();	
				}	
				
				$order_id  = $order->get_id();

				$update_send_sms = absint($data['update_send_sms']);
				$message = '';

				if( $update_send_sms == 1 or $update_send_sms == 0){

					$order->update_meta_data( 'iben_sms_lexicon_woo_customer_order_notification_meta', $update_send_sms);	
					$order->save();
					$message = esc_html__('Updated', 'iben-woo-elta-courier');
					
				}

				$resultss = array (
					"status" => "success",
					"result" => $message,
					"update_send_sms" => $update_send_sms
				);
		
				echo json_encode ( $resultss );
					
			}
		
			// ajax handlers must die
			wp_die();
		}		
		




		/**
		 *
		 * @since     1.0.0
		 */
		public function iben_elta_meta_box_html($order, $order_list = false) { 

			if ( !is_a( $order, 'WC_Order' )  ){
				return;
			}

			$order_id  = $order->get_id();
			$iben_elta_meta_values = $order->get_meta( '_iben_elta_voucher_data', true) ;

			ob_start(); 

			$main_settings = $this->main_settings;

			if ( defined( 'WC_VERSION' ) && version_compare( WC_VERSION, '6.9.0', '>=' ) ) {
				$getpost_type = OrderUtil::get_order_type( $order_id );
			}else{
				$getpost_type = get_post_type( $order_id);
			}

			?>

			<input type="hidden" id="iben_elta_order_metabox_post_type" name="iben_elta_order_metabox_post_type" value="<?php echo esc_attr( $getpost_type ); ?>" size="25" />
			<input type="hidden" id="iben_elta_order_metabox_order_id" name="iben_elta_order_metabox_order_id" value="<?php echo esc_attr( $order_id ); ?>" size="25" />
			
			<?php 
			$order_page = ($order_list ? 'list' : 'order'); 
			$order_list = ($order_list ? 'true' : 'false');
			?>
			<input type="hidden" id="iben_elta_order_metabox_position_page" name="iben_elta_order_metabox_position_page" value="<?php echo esc_attr( $order_page ); ?>" size="25" />

			<p class="iben_elta_convert_uppercase" style="margin: 0px;"><a href="#" id="iben_elta_convert_uppercase" data-list = "<?php echo esc_attr($order_list); ?>" data-id_order="<?php echo esc_attr( $order_id ); ?>" ><?php esc_html_e('Convert to UPPERCASE', 'iben-woo-elta-courier'); ?></a></p>
			
			<div class="iben_elta_order_inner_inside">
			
				<div class="iben-elta-left">
						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_fullname">
								<b><?php esc_html_e( 'Full name', 'iben-woo-elta-courier' ); ?>*</b>
							</label>
							<?php 
							
							$first_name = '';
							if( $order->get_billing_first_name() != $order->get_shipping_first_name() && !empty($order->get_shipping_first_name()) ) {
								$first_name = $order->get_shipping_first_name();
							} else {
								$first_name = $order->get_billing_first_name();
							}
							
							$last_name = '';
							if( $order->get_billing_last_name() != $order->get_shipping_last_name() && !empty($order->get_shipping_last_name()) ) {
								$last_name = $order->get_shipping_last_name();
							} else {
								$last_name = $order->get_billing_last_name();
							}
							
							$company = '';
							if( $order->get_billing_company() != $order->get_shipping_company() && !empty($order->get_shipping_company()) ) {
								$company = $order->get_shipping_company();
							} else {
								$company = $order->get_billing_company();
							}
							$company = ( $company == '' )? '':' ('.$company.')';
							

							$fullname = $last_name.' '.$first_name.$company; 
							$fullname = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($fullname, 'UTF-8') : $fullname;

							$fullname = apply_filters( 'iben_elta_filter_fullname_order_metabox', $fullname, $order_id );

							if ( isset($iben_elta_meta_values['fullname']) && $iben_elta_meta_values['fullname'] != '' && $iben_elta_meta_values['fullname'] != $fullname ){
								$fullname = $iben_elta_meta_values['fullname'];
							}
							
							?>
							<input type="text" id="iben_elta_order_metabox_fullname" class="iben_elta_order_metabox_fullname" name="iben_elta_order_metabox_fullname" value="<?php echo esc_attr( $fullname ); ?>" size="25" />
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_address">
								<b><?php esc_html_e( 'Address', 'iben-woo-elta-courier' ); ?>*</b>
							</label>
							<?php 
							$address_1 = '';
							$address_2 = '';
							if( $order->get_billing_address_1() != $order->get_shipping_address_1() && !empty($order->get_shipping_address_1()) ) {
								$address_1 = $order->get_shipping_address_1();
							} else {
								$address_1 = $order->get_billing_address_1();
							}


							if( $order->get_billing_address_2() != $order->get_shipping_address_2() && !empty($order->get_shipping_address_2()) ) {
								$address_2 = $order->get_shipping_address_2();
							} else {
								$address_2 = $order->get_billing_address_2();
							}

							$comma = ( $address_2 == '' )? '':',';
							$address = $address_1.$comma.' '.$address_2; 
							
							$address = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($address, 'UTF-8') : $address;

							$address = apply_filters( 'iben_elta_filter_address_order_metabox', $address, $order_id );

							if ( isset($iben_elta_meta_values['address']) && $iben_elta_meta_values['address'] != '' && $iben_elta_meta_values['address']!= $address ){
								$address = $iben_elta_meta_values['address'];
							}
							
							?>
							<input type="text" id="iben_elta_order_metabox_address" class="iben_elta_order_metabox_address" name="iben_elta_order_metabox_address" value="<?php echo esc_attr( $address ); ?>" size="25" />
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_postalcode">
								<b><?php esc_html_e( 'Postal code', 'iben-woo-elta-courier' ); ?>*</b>
							</label>
							<?php 
							$postcode = '';
							if( $order->get_billing_postcode() != $order->get_shipping_postcode() && !empty($order->get_shipping_postcode())  ) {
								$postcode = $order->get_shipping_postcode();
							} else {
								$postcode = $order->get_billing_postcode();
							}

							$postcode = apply_filters( 'iben_elta_filter_postcode_order_metabox', $postcode, $order_id );

							if (isset($iben_elta_meta_values['postal_code']) &&  $iben_elta_meta_values['postal_code'] != '' && $iben_elta_meta_values['postal_code']!= $postcode ){
								$postcode = $iben_elta_meta_values['postal_code'];
							}

							?>
							<input type="text" id="iben_elta_order_metabox_postalcode" class="iben_elta_order_metabox_postalcode" name="iben_elta_order_metabox_postalcode" value="<?php echo esc_attr( $postcode ); ?>" size="25" />
							<span class="iben_elta_order_postalcode_field_station"></span>
						</p>


						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_region">
								<b><?php esc_html_e( 'Region', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php

							$billing_country = $order->get_billing_country();
							$shipping_country = $order->get_shipping_country();


							if ( $billing_country != $shipping_country && $shipping_country != '' ){
								$country = $shipping_country;
							}else{
								$country = $billing_country;
							}


							$billing_state = $order->get_billing_state();
							$shipping_state = $order->get_shipping_state();

							if ( $billing_state != $shipping_state && $shipping_state != '' ){
								$state = $shipping_state;
							}else{
								$state = $billing_state;
							}

							if ($country){
								if ($state){
									$state =  WC()->countries->get_states( $country )[$state];
								}else{
									$state = '';
								}
								
							}else{

								if ($state){
									$state =  WC()->countries->get_states( 'GR' )[$state];
								}else{
									$state = '';
								}

							}
							
							
							if ($order->get_billing_city() != $order->get_shipping_city() && !empty($order->get_shipping_city()) ){
								$city = $order->get_shipping_city();
							}else{
								$city = $order->get_billing_city();
							}

							$region = $city.', '.$state; 

							$region = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($region, 'UTF-8') : $region;
							
							$region = apply_filters( 'iben_elta_filter_region_order_metabox', $region, $order_id );

							if ( isset($iben_elta_meta_values['region']) && $iben_elta_meta_values['region'] != '' && $iben_elta_meta_values['region'] != $region ){
								$region = $iben_elta_meta_values['region'];
							}
							
							?>
							<input type="text" id="iben_elta_order_metabox_region" class="iben_elta_order_metabox_region" name="iben_elta_order_metabox_region" value="<?php echo esc_attr( $region ); ?>" size="25" />
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_phone1">
								<b><?php esc_html_e( 'Phone 1', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php 
									
							$billing_phone1 = apply_filters('iben_elta_phone_format', $order->get_billing_phone(), $order_id);
							if ($billing_phone1 !== null) {
								$billing_phone1 = str_replace(array(' ', '-', '.', '+'), '', $billing_phone1 ?? '');
							}

							$shipping_phone1 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_shipping_phone1_metafield_order_metabox', 'shipping_phone' ), true));
							$shipping_phone1 = apply_filters('iben_elta_phone_format', $shipping_phone1, $order_id);
							if ($shipping_phone1 !== null) {
								$shipping_phone1 = str_replace(array(' ', '-', '.', '+'), '', $shipping_phone1 ?? '');
							}							

							if ( $billing_phone1 != $shipping_phone1 && $shipping_phone1 != ''){
								$phone1 = $shipping_phone1;
							}else{
								$phone1 = $billing_phone1;
							}
							
							$phone1 = apply_filters( 'iben_elta_filter_phone1_order_metabox', $phone1, $order_id );

							if ( isset($iben_elta_meta_values['phone1']) && $iben_elta_meta_values['phone1'] != $phone1 ){
								$phone1 = $iben_elta_meta_values['phone1'];
							}
							

							?>
							<input type="text" id="iben_elta_order_metabox_phone1" class="iben_elta_order_metabox_phone1" name="iben_elta_order_metabox_phone1" value="<?php echo esc_attr( $phone1 ); ?>" size="25" />
						</p>
						
						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_phone2">
								<b><?php esc_html_e( 'Phone 2', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php 
							
							$billing_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_phone2_metafield_order_metabox', 'billing_mobile' ), true));
							$billing_phone2 = apply_filters('iben_elta_phone_format', $billing_phone2, $order_id);

							$shipping_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_shipping_phone2_metafield_order_metabox', 'shipping_mobile' ), true));
							$shipping_phone2 = apply_filters('iben_elta_phone_format', $shipping_phone2, $order_id);
			
							if ( $billing_phone2 != $shipping_phone2 && $shipping_phone2 != '' ){
								$phone2 = $shipping_phone2;
							}else{
								$phone2 = $billing_phone2;
							}

							$phone2 = apply_filters( 'iben_elta_filter_phone2_order_metabox', $phone2, $order_id );

							if ( isset($iben_elta_meta_values['phone2']) && $iben_elta_meta_values['phone2'] != $phone2 ){
								$phone2 = $iben_elta_meta_values['phone2'];
							}

							?>
							<input type="text" id="iben_elta_order_metabox_phone2" class="iben_elta_order_metabox_phone2" name="iben_elta_order_metabox_phone2" value="<?php echo esc_attr( $phone2 ); ?>" size="25" />
						</p>
						
						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_comments">
								<b><?php esc_html_e( 'Comments', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php
							$comments = $order->get_customer_note();
							
							$comments = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($comments, 'UTF-8') : $comments;
							
							$comments = apply_filters( 'iben_elta_filter_comments_order_metabox', $comments, $order_id );
							
							if ( isset($iben_elta_meta_values['comments']) && $iben_elta_meta_values['comments'] != $comments ){
								$comments = $iben_elta_meta_values['comments'];
							}

							?>
							<textarea id="iben_elta_order_metabox_comments" class="iben_elta_order_metabox_comments" name="iben_elta_order_metabox_comments"><?php echo esc_attr( $comments ); ?></textarea>
							<span style="display: block;"><span id="iben_elta_order_metabox_comments_counter"></span>/ 57 <?php esc_html_e( 'Characters', 'iben-woo-elta-courier' ); ?></span>
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_cash_delivery">
								<b><?php esc_html_e( 'Cash on delivery', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php 
							
							$payment = $order->get_payment_method();

							//compatible with WC cod and 3rd party COD with name as id_codfee
							$payments = array('cod', 'id_codfee');

							$payments_filter = array();
							$payments_filter = apply_filters( 'iben_elta_filter_cashdelivery_payments_order_metabox', $payments_filter );
							
							if(is_array($payments_filter) && !empty($payments_filter)){
								foreach ($payments_filter as $payment_filter){
									$payments[] = $payment_filter;
								}
							}


							if (in_array($payment, $payments, true)) {
								$cashdelivery = $order->get_total();	
							}else{
								$cashdelivery = '0'; 
							}
							$cashdelivery = apply_filters( 'iben_elta_filter_cashdelivery_order_metabox', $cashdelivery, $order_id );

							if ( isset($iben_elta_meta_values['cash_delivery']) && $iben_elta_meta_values['cash_delivery'] != $cashdelivery ){
								$cashdelivery = $iben_elta_meta_values['cash_delivery'];
							}
							
							?>
							<input type="text" id="iben_elta_order_metabox_cash_delivery" class="iben_elta_order_metabox_cash_delivery" name="iben_elta_order_metabox_cash_delivery" value="<?php echo esc_attr( $cashdelivery ); ?>" size="25" />
						</p>	

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_weight">
								<b><?php esc_html_e( 'Weight', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php
							
							// check if default weight has been set and use this
							$default_weight = sanitize_text_field($main_settings['iben_elta_main_settings_default_weight']);
							if( isset( $default_weight ) && $default_weight > 0 ) {
								
								$weight = $default_weight;

							}else{
						
								$total_weight = 0;
								foreach( $order->get_items() as $item_id => $product_item ){
									$quantity = $product_item->get_quantity(); // get quantity
									$product = $product_item->get_product(); // get the WC_Product object
									if( $product ) {
										$product_weight = $product->get_weight(); // get the product weight
										$total_weight += floatval( ( $product_weight ? $product_weight : 0 ) * $quantity );
									  }
									
								}
								
								$weight = round($total_weight, 3);
								

								$unit = get_option('woocommerce_weight_unit');

								if ($unit == 'g'){
									$weight = $weight/1000;
								}
								
							}
							//check if has saved voucher info
							$weight = apply_filters( 'iben_elta_filter_weight_order_metabox', $weight, $order_id );

							if ( isset($iben_elta_meta_values['weight']) && $iben_elta_meta_values['weight'] != '' && $iben_elta_meta_values['weight'] != $weight ){
								$weight = $iben_elta_meta_values['weight'];
							}
							
							?>
							<input type="text" id="iben_elta_order_metabox_weight" class="iben_elta_order_metabox_weight" name="iben_elta_order_metabox_weight" value="<?php echo esc_attr( $weight ); ?>" size="25" />
						</p>				

						<div><a href='#' id ='iben_elta_order_create_voucher_more'><?php esc_html_e( 'More options', 'iben-woo-elta-courier' ); ?></a></div>

						<div class="iben_elta_order_loadmore_hidden_section">

							<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_packages_number">
									<b><?php esc_html_e( 'Number of parcels', 'iben-woo-elta-courier' ); ?></b>
								</label>
								<?php 
								$parcels = '1'; 
								
								if ( isset($iben_elta_meta_values['packages_number']) && $iben_elta_meta_values['packages_number'] != '' && $iben_elta_meta_values['packages_number'] != $parcels ){
									$parcels = $iben_elta_meta_values['packages_number'];
								}
								?>
								<input type="text" id="iben_elta_order_metabox_packages_number" class="iben_elta_order_metabox_packages_number" name="iben_elta_order_metabox_packages_number" value="<?php echo esc_attr( $parcels ); ?>" size="25" />
							</p>

							<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_ref">
									<b><?php esc_html_e( 'Reference No', 'iben-woo-elta-courier' ); ?></b>
								</label>
								<?php
								
								$ref = '';

								if (isset( $main_settings['iben_elta_main_settings_enable_ref_order'] ) ){

									$ref = $order->get_order_number();
								}
								$ref = apply_filters( 'iben_elta_filter_ref_order_metabox', $ref, $order_id );
								
								if ( isset($iben_elta_meta_values['ref']) && $iben_elta_meta_values['ref'] != '' && $iben_elta_meta_values['ref'] != $ref ){
									$ref = $iben_elta_meta_values['ref'];
								}
								
								?>
								<input type="text" id="iben_elta_order_metabox_ref" class="iben_elta_order_metabox_ref" name="iben_elta_order_metabox_ref" value="<?php echo esc_attr( $ref ); ?>" size="25" />
							</p>

							<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_service">
									<b><?php esc_html_e( 'Service', 'iben-woo-elta-courier' ); ?></b>
								</label>

								<?php
									if ( isset($iben_elta_meta_values['service']) && $iben_elta_meta_values['service'] != '' ){
										$service = $iben_elta_meta_values['service'];
									}else{
										$service = 1;
									}							
								?>
								<select id="iben_elta_order_metabox_service" name="iben_elta_order_metabox_service">
									<option value="1" <?php echo ($service == '1' ? 'selected' : ''); ?> >1 - 201</option>
									<option value="2" <?php echo ($service == '2' ? 'selected' : ''); ?> >2 - 204 </option>
								</select>
							</p>

							<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_subcode">
									<b><?php esc_html_e( 'Subcode', 'iben-woo-elta-courier' ); ?></b>
								</label>
								<?php

								$subcode = ''; 

								$display_subcode_from_user = apply_filters( 'iben_elta_filter_automatic_display_subcode_from_user', true);

								//dokan compatibility here
								if (function_exists('dokan_get_seller_id_by_order') && $display_subcode_from_user) {	
									$seller_id = dokan_get_seller_id_by_order( $order_id );	
									
									$user_subcode = get_user_meta($seller_id, 'iben_elta_user_subcode_meta', true);				
									
									if ($user_subcode != '' ){
										$subcode = $user_subcode;
									}
								}

		
								
								if ( isset($iben_elta_meta_values['subcode']) && $iben_elta_meta_values['subcode'] != '' && $iben_elta_meta_values['subcode'] != $subcode ){
									$subcode = $iben_elta_meta_values['subcode'];
								}

								?>
								<input type="text" id="iben_elta_order_metabox_subcode" class="iben_elta_order_metabox_subcode" name="iben_elta_order_metabox_subcode" value="<?php echo esc_attr( $subcode ); ?>" size="25" />
							</p>

							<?php if ($order_list == 'false'){ ?>
							<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_sur_1">
									<b><?php esc_html_e( 'Special management', 'iben-woo-elta-courier' ); ?></b>
								</label>
								<?php
									if ( isset($iben_elta_meta_values['sur_1']) && $iben_elta_meta_values['sur_1'] != '' ){
										$sur_1 = $iben_elta_meta_values['sur_1'];
									}else{
										$sur_1 = 0;
									}				
								?>								
						
								<select id="iben_elta_order_metabox_sur_1" name="iben_elta_order_metabox_sur_1" data-list = "<?php echo esc_attr($order_list); ?>" data-id_order="<?php echo esc_attr( $order_id ); ?>" >
									<option value="0" <?php echo ($sur_1 == 0 ? 'selected' : ''); ?> ><?php esc_html_e( 'No', 'iben-woo-elta-courier' ); ?></option>
									<option value="1" <?php echo ($sur_1 == 1 ? 'selected' : ''); ?> ><?php esc_html_e( 'Yes', 'iben-woo-elta-courier' ); ?></option>
								</select>
							</p>
							<?php } ?>

							<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_sur_2">
									<b><?php esc_html_e( 'Delivery time', 'iben-woo-elta-courier' ); ?></b>
								</label>
								<?php
									if ( isset($iben_elta_meta_values['sur_2']) && $iben_elta_meta_values['sur_2'] != '' ){
										$sur_2 = $iben_elta_meta_values['sur_2'];
									}else{
										$sur_2 = 0;
									}				
								?>							
								<select id="iben_elta_order_metabox_sur_2" name="iben_elta_order_metabox_sur_2">
									<option value="0" <?php echo ($sur_2 == 0 ? 'selected' : ''); ?>><?php esc_html_e( 'No', 'iben-woo-elta-courier' ); ?></option>
									<option value="1" <?php echo ($sur_2 == 1 ? 'selected' : ''); ?>><?php esc_html_e( 'Yes', 'iben-woo-elta-courier' ); ?></option>
								</select>
							</p>

							<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_sur_3">
									<b><?php esc_html_e( 'Delivery Saturday', 'iben-woo-elta-courier' ); ?></b>
								</label>
								<?php
									if ( isset($iben_elta_meta_values['sur_3']) && $iben_elta_meta_values['sur_3'] != '' ){
										$sur_3 = $iben_elta_meta_values['sur_3'];
									}else{
										$sur_3 = 0;
									}				
								?>							
								<select id="iben_elta_order_metabox_sur_3" name="iben_elta_order_metabox_sur_3">
									<option value="0" <?php echo ($sur_3 == 0 ? 'selected' : ''); ?>><?php esc_html_e( 'No', 'iben-woo-elta-courier' ); ?></option>
									<option value="1" <?php echo ($sur_3 == 1 ? 'selected' : ''); ?>><?php esc_html_e( 'Yes', 'iben-woo-elta-courier' ); ?></option>
								</select>
							</p>
						</div>			
					
					<p class="iben_elta-field-buttons iben-normal-shipping-button"> 
						<?php 
						
							$qry = iben_elta_get_Vouchers_from_id_order( $order_id );
							
							//iben *Geniki compatibility
							$geniki_voucher_exist = false;
							if ( class_exists( 'Iben_Woo_Geniki' ) && function_exists('iben_geniki_get_Vouchers_from_id_order') ) {

								$qrys = iben_geniki_get_Vouchers_from_id_order($order_id);
			
								if( $qrys ){
									$geniki_voucher_exist = true;
								}
			
							}

							//iben *ACS compatibility
							$acs_voucher_exist = false;
							if ( class_exists( 'Iben_Woo_Acs' ) && function_exists('iben_acs_get_Vouchers_from_id_order') ) {

								$qry_Acs = iben_acs_get_Vouchers_from_id_order($order_id);

								if( $qry_Acs ){
									$acs_voucher_exist = true;
								}

							}

							//iben *Speedex compatibility
							$speedex_voucher_exist = false;
							if ( class_exists( 'Iben_Woo_Speedex' ) && function_exists('iben_speedex_get_Vouchers_from_id_order') ) {

								$qrys = iben_speedex_get_Vouchers_from_id_order($order_id);
			
								if( $qrys ){
									$speedex_voucher_exist = true;
								}
			
							}  														
							
							if ($qry || $geniki_voucher_exist || $acs_voucher_exist || $speedex_voucher_exist){

							?>
							<span class="iben_elta_confirm_duplicatedvoucher">
								<input type="checkbox" id="iben_elta_confirm_duplicatedvoucher" name="" value="1" /><?php esc_html_e( 'Confirm by checking here if you want to create another voucher without delete previous voucher', 'iben-woo-elta-courier' ); ?>
							</span>
							<?php } ?>

						<button type="button" id="iben_elta_order_create_voucher" data-list = "<?php echo esc_attr($order_list); ?>" data-id_order = "<?php echo absint( $order_id ); ?>" class="button button-primary"><?php esc_html_e( 'Create voucher (SYDETA)', 'iben-woo-elta-courier' ); ?></button>
						<span id="iben_elta_order_create_voucher_response_<?php echo absint( $order_id ); ?>" style="display: none;"></span>
					</p>
					<p class="iben_elta-field-buttons iben-returned-shipping-button" style="display: none;"> 
							<?php 
							$qry = iben_elta_get_ReturnedVouchers_from_id_order( $order_id );
							if ($qry){
							?>
							<span class="iben_elta_confirm_duplicatedvoucher">
								<input type="checkbox" id="iben_elta_confirm_duplicated_returned_voucher" name="" value="1" /><?php esc_html_e( 'Confirm by checking here if you want to create another returned voucher without delete previous voucher', 'iben-woo-elta-courier' ); ?>
							</span>
							<?php } ?>

						<button type="button" id="iben_elta_order_create_returned_voucher" data-list = "<?php echo esc_attr($order_list); ?>" data-id_order = "<?php echo absint( $order_id );?>" class="button button-primary"><?php esc_html_e( 'Create Returned Voucher (SYDETA)', 'iben-woo-elta-courier' ); ?></button>
					</p>


					<?php 
					if( isset( $main_settings['iben_elta_main_settings_2nd_enable_account'] ) ) {

							$account_selected = apply_filters( 'iben_elta_filter_order_elta_account', 1, $weight, $order_id );
						
						?> 
						<span style="display: block; padding-top: 5px;"> 
							<select id="iben_elta_order_metabox_elta_account" name="iben_elta_order_metabox_elta_account">
								<option value="1" <?php echo ($account_selected == 1 ? 'selected' : ''); ?> ><?php echo apply_filters( 'iben_elta_filter_order_elta_1st_account_name', esc_html__( '1st account', 'iben-woo-elta-courier' )); ?></option>
								<option value="2" <?php echo ($account_selected == 2 ? 'selected' : ''); ?> ><?php echo apply_filters( 'iben_elta_filter_order_elta_2nd_account_name', esc_html__( '2nd account', 'iben-woo-elta-courier' )); ?></option>
							</select>
						</span>
					<?php } ?>	
						
						<?php						
						echo "<div class='iben_elta_settings_info_button'>";
						if( isset( $main_settings['iben_elta_main_settings_email_completed'] ) || isset( $main_settings['iben_elta_button_settings_create_send_completed_email'] ) || isset( $main_settings['iben_elta_button_settings_create_pdf'] ) ) {
							echo '<span class="iben_elta_settings_info_button_title">'.esc_html__( 'According your settings, when you create the voucher:', 'iben-woo-elta-courier' ).'</span>';
						}
						echo "<ul>";



						if( isset( $main_settings['iben_elta_button_settings_create_send_completed_email'] ) ) {
							$checkbox_send_completed = '<input type="checkbox" class="iben_elta_create_voucher_settings" id="iben_elta_button_settings_create_send_completed_email" name="iben_elta_main_settings_fields[iben_elta_button_settings_create_send_completed_email]" value="1" ' . checked( 1, isset( $main_settings['iben_elta_button_settings_create_send_completed_email'] ) ? $main_settings['iben_elta_button_settings_create_send_completed_email'] : 0, false ) . '/>';
							echo '<li>'.esc_html__( 'Completed email will be forced sent', 'iben-woo-elta-courier' ) . $checkbox_send_completed .' | </li>';
						}
						
						if( isset( $main_settings['iben_elta_main_settings_voucher_change_order_status'] ) && $main_settings['iben_elta_main_settings_voucher_change_order_status'] != -1) {
							
							$html = iben_elta_get_order_status_text($main_settings['iben_elta_main_settings_voucher_change_order_status']);								
							echo '<li>'.esc_html__( 'Order status will be changed to', 'iben-woo-elta-courier' ).' <b>'. $html .'</b> | </li>';
						}
						if( isset( $main_settings['iben_elta_button_settings_create_pdf'] ) ) {
							$checkbox_create_pdf = '<input type="checkbox" class="iben_elta_create_voucher_settings" id="iben_elta_button_settings_create_pdf" name="iben_elta_main_settings_fields[iben_elta_button_settings_create_pdf]" value="1" ' . checked( 1, isset( $main_settings['iben_elta_button_settings_create_pdf'] ) ? $main_settings['iben_elta_button_settings_create_pdf'] : 0, false ) . '/>';
							echo '<li>'.esc_html__( 'PDF will be generated immediately', 'iben-woo-elta-courier' ). $checkbox_create_pdf .'</li>';
						}

						if( isset( $main_settings['iben_elta_main_settings_email_completed'] ) ) {

							$is_default = isset( $main_settings['iben_elta_main_settings_email_checked_default_completed'] ) ? (int)$main_settings['iben_elta_main_settings_email_checked_default_completed'] : 0;	
				
							$checked = '';
				
							if ( $order->get_meta( 'iben_elta_tracking_order_notification', true) == 1 ){
								$checked = 'checked="checked"';
							}else{
								if ( $is_default && !$order->meta_exists('iben_elta_tracking_order_notification') ){
									$checked = 'checked="checked"';
								}
							}

							$checkbox_tracking_on_completed = '<input type="checkbox" class="iben_elta_create_voucher_settings" id="iben_elta_main_settings_email_completed" name="iben_elta_main_settings_fields[iben_elta_main_settings_email_completed]" value="1" ' . $checked . '/>';
							echo '<li>'.esc_html__( 'Include tracking info on completed email', 'iben-woo-elta-courier' ). $checkbox_tracking_on_completed .' | </li>';
						}


						if (class_exists('Iben_Sms_Lexicon_Admin_Woo')) {

							$options = get_option( 'iben_sms_lexicon_woo_settings' );	

							$notification_order_state = isset( $options['iben_sms_lexicon_woo_customer_notification_order_state_enable'] ) ? (int)$options['iben_sms_lexicon_woo_customer_notification_order_state_enable'] : 0;
							
							$is_default = isset( $options['completed_checkbox'] ) ? $options['completed_checkbox'] : 0;	

							if ($notification_order_state && $is_default){

								$checked = '';
								if ( $order->meta_exists('iben_sms_lexicon_woo_customer_order_notification_meta') ){
									if ( $order->get_meta( 'iben_sms_lexicon_woo_customer_order_notification_meta', true) == 1 ){
										$checked = 'checked="checked"';
									}
								}else{
									if ( !isset($options['iben_sms_lexicon_woo_customer_notification_checkbox_enable']) ){
										$checked = 'checked="checked"';									
									}					
								}

								$checkbox_send_iben_sms = '<input type="checkbox" class="iben_elta_create_voucher_settings" id="iben_elta_button_settings_send_iben_sms_completed" name="iben_elta_button_settings_send_iben_sms_completed" value="1" ' . $checked . '/>';
								echo '<li>'.esc_html__( 'Send completed SMS', 'iben-woo-elta-courier' ). $checkbox_send_iben_sms .'</li>';

							}
						}						


						echo "</ul> </div>";
						?>
					



				</div>

				<div class="iben-elta-right">
						<div class="iben_elta_order_right_box_vouchers">
							<h3 class="iben_elta_order_box_title2"><?php esc_html_e( 'Vouchers', 'iben-woo-elta-courier' ); ?></h3>

							<div class='iben_elta_order_vouchers'>

								<span style="display: inline-block; padding: 5px 10px; vertical-align: top;">

									<?php

									$regenerate = false;

									$get_vouchers = iben_elta_get_Vouchers_from_id_order($order_id);
									
									$display = ( (!empty($get_vouchers) && sizeof($get_vouchers) != 0 ) ? 'block' : 'none');
									echo '<span id="iben_elta_order_vouchers_title_response" style="display: '.$display.';">'.esc_html__( 'Normal shipping', 'iben-woo-elta-courier' ).'</span>';
									?>
									<span id="iben_elta_order_voucher_response" class="iben_elta_order_voucher_response">
									<?php
									
									if (!empty($get_vouchers) && sizeof($get_vouchers) != 0 ){
										foreach ($get_vouchers as $voucher){
											
											echo $this->iben_display_html_vouchers($voucher->time, $voucher->elta_delivery_vouchers, $order_id, false, false);

										}

										$regenerate = true;
									}

									$get_compined = iben_elta_get_Compined_Voucher_from_id_order($order_id);
									

									if ($get_compined){

										$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$get_compined.".pdf";
										if(is_file($file)){
											echo iben_elta_Compined_Voucher_Link($get_compined);
											
										}
										
									}

									

									?>
									</span>

									<span style="display: block;" id="iben_elta_compine_vouchers_respond"> </span>
									<span style="display: inline-flex;align-content: center;align-items: center;">


									<button style="display: <?php echo esc_attr($display); ?>;" data-list = "<?php echo esc_attr($order_list); ?>" data-id_order = "<?php echo absint( $order_id );?>" type="button" class="button iben_elta_order_button_compine_vouchers button-secondary"><?php ($regenerate ? esc_html_e( 'Re generate PDF', 'iben-woo-elta-courier' ) : esc_html_e( 'Generate and print PDF', 'iben-woo-elta-courier' )) ?></button>
									<div class="iben_elta_compine_vouchers_spinner iben_elta_spinners" style="display:none;"><div class="bounce1"></div><div class="bounce2"></div><div class="bounce3"></div></div>

								</span>

								</span>
							
								<span style="display: inline-block; padding: 5px 10px; vertical-align: top;">
									<?php	
										$regenerate = false;

										$get_returned_vouchers = iben_elta_get_ReturnedVouchers_from_id_order($order_id);
										
										$display = ( (!empty($get_returned_vouchers) && sizeof($get_returned_vouchers) != 0 ) ? 'block' : 'none');
										echo '<span id="iben_elta_order_returned_vouchers_title_response" style="display: '.$display.'; ">'.esc_html__( 'Special management', 'iben-woo-elta-courier' ).'</span>';
					
										?>
										<span id="iben_elta_order_returned_vouchers_response" class="iben_elta_order_returned_vouchers_response">
										<?php
										if (!empty($get_returned_vouchers) && sizeof($get_returned_vouchers) != 0 ){
											foreach ($get_returned_vouchers as $voucher){
												
												echo $this->iben_display_html_vouchers($voucher->time, $voucher->elta_returned_vouchers, $order_id);

											}
											$regenerate = true;
										}

										$get_compined = iben_elta_get_Compined_Returned_Voucher_from_id_order($order_id);

										if ($get_compined){

											$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$get_compined.".pdf";
											if(is_file($file)){
												echo iben_elta_Compined_Voucher_Link($get_compined);
											}
											
										}
									?>
									</span>
									<span style="display: block;" id="iben_elta_compine_returned_vouchers_respond"> </span>
									<span style="display: inline-flex;align-content: center;align-items: center;">
										<button style="display: <?php echo esc_attr($display); ?>;" type="button" class="button iben_elta_order_button_compine_ret_vouchers button-secondary"><?php ($regenerate ? esc_html_e( 'Re generate PDF', 'iben-woo-elta-courier' ) : esc_html_e( 'Generate and print PDF', 'iben-woo-elta-courier' )) ?></button>
										<div class="iben_elta_compine_returned_vouchers_spinner iben_elta_spinners" style="display:none;"><div class="bounce1"></div><div class="bounce2"></div><div class="bounce3"></div></div>
									</span>
								</span>
							
							</div>
						</div>

						<div class="iben_elta_order_right_box_tools">
							<h3 class="iben_elta_order_box_title2"><?php esc_html_e( 'Tools', 'iben-woo-elta-courier' ); ?></h3>
							<div style="display: block;">
								<?php

									$order_tracking_number = $order->get_meta( '_iben_elta_voucher_ship_number', true);
								?>
								<div style="display: block; padding: 5px 5px; text-align: left;"> 
								<select style="width: 100px;" id="iben_elta_order_tracing_voucher_ref" name="iben_elta_order_tracing_voucher_ref" class="list_tracing_ref<?php echo absint( $order->get_id() ); ?>" >
										<option value="1" ><?php esc_html_e( 'By Voucher', 'iben-woo-elta-courier' ); ?></option>
										<option value="2" ><?php esc_html_e( 'By ref', 'iben-woo-elta-courier' ); ?></option>
								</select>
									<input style="width: 100px;"  type="text" id="iben_elta_order_tracing_voucher" class="list_tracing_tracing_voucher<?php echo absint( $order->get_id() ); ?>"  value=<?=$order_tracking_number;?> >
									<button style="" type="button" id="iben_elta_order_tracing_button" class="button iben_elta_order_tracing_button list_tracing_button<?php echo absint( $order->get_id() ); ?>  button-secondary" data-list = "<?php echo esc_attr($order_list); ?>" data-id_order = "<?php echo absint( $order_id ); ?>"><?php esc_html_e( 'Tracking voucher', 'iben-woo-elta-courier' ); ?></button>
								</div>
								
								<div style="display: block; padding: 5px 5px; text-align: left;"> 

									<input type="text" id="iben_elta_order_station_voucher" class="list_station_zip<?php echo absint( $order->get_id() ); ?>"  placeholder = '<?php esc_html_e( 'Search ELTA Courier with ZIP', 'iben-woo-elta-courier' ); ?>' value="<?php echo sanitize_text_field( $order->get_billing_postcode() ); ?>">
									<button style="" type="button" id="iben_elta_order_station_button" class="button iben_elta_order_station_button list_station_button<?php echo absint( $order->get_id() ); ?>  button-secondary" data-list = "<?php echo esc_attr($order_list); ?>" data-id_order = "<?php echo absint( $order_id ); ?>"><?php esc_html_e( 'Check Station', 'iben-woo-elta-courier' ); ?></button>
								</div>
							</div>

							<span id="iben_elta_order_tools_response" class='iben_elta_order_box_tools_inner tools_response_id_order_<?php echo absint( $order->get_id() ); ?>'></span>							
						</div>

				</div>		
			</div>

			<?php
			return ob_get_clean();
		}

		

		/**
		 *
		 * @since     1.3.0
		 */
		public function iben_elta_meta_box_html_single() { 
			
			ob_start(); 
			$main_settings = $this->main_settings;
			?>

			<input type="hidden" id="iben_elta_order_metabox_order_id_single" name="iben_elta_order_metabox_order_id_single" value="-1" size="25" />	

			<p class="iben_elta_convert_single_uppercase" style="margin: 0px;"><a href="#" id="iben_elta_convert_single_uppercase"><?php esc_html_e('Convert to UPPERCASE', 'iben-woo-elta-courier'); ?></a></p>
			

			<div id="iben_elta_contact_save_response" style="font-weight: bold;"></div>
				
			<div class="iben_elta_order_inner_inside">
			
				<div class="iben-elta-left">
					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_id_referrence_single">
							<b><?php esc_html_e( 'Code reference', 'iben-woo-elta-courier' ); ?>*</b>
						</label>
						<input type="text" id="iben_elta_order_metabox_id_referrence_single" class="iben_elta_order_metabox_id_referrence_single" name="iben_elta_order_metabox_id_referrence_single" value="" size="20" />
						
						<?php echo '<a href="' . admin_url( 'admin.php?page=iben_woo_elta_contacts' ) . '" style="display: block;font-size: 11px;">'. esc_html__('Contacts list', 'iben-woo-elta-courier') .'</a> '; ?>
					</p>

					<p class="iben-elta-field">
						<a href="#" id="iben_elta_contact_load" style="padding-bottom: 5px;"><?php esc_html_e('Load contact', 'iben-woo-elta-courier'); ?></a>
						<a href="#" id="iben_elta_contact_save"><?php esc_html_e('Save / Update', 'iben-woo-elta-courier'); ?></a>
					</p>

					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_fullname_single">
							<b><?php esc_html_e( 'Full name', 'iben-woo-elta-courier' ); ?>*</b>
						</label>
						<input type="text" id="iben_elta_order_metabox_fullname_single" class="iben_elta_order_metabox_fullname_single" name="iben_elta_order_metabox_fullname_single" value="" size="25" />
					</p>

					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_address_single">
							<b><?php esc_html_e( 'Address', 'iben-woo-elta-courier' ); ?>*</b>
						</label>

						<input type="text" id="iben_elta_order_metabox_address_single" class="iben_elta_order_metabox_address_single" name="iben_elta_order_metabox_address_single" value="" size="25" />
					</p>
					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_region_single">
							<b><?php esc_html_e( 'Region', 'iben-woo-elta-courier' ); ?></b>
						</label>
						<input type="text" id="iben_elta_order_metabox_region_single" class="iben_elta_order_metabox_region_single" name="iben_elta_order_metabox_region_single" value="" size="25" />
					</p>

					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_postalcode_single">
							<b><?php esc_html_e( 'Postal code', 'iben-woo-elta-courier' ); ?>*</b>
						</label>
	
						<input type="text" id="iben_elta_order_metabox_postalcode_single" class="iben_elta_order_metabox_postalcode_single" name="iben_elta_order_metabox_postalcode_single" value="" size="25" />

					</p>

					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_phone1_single">
							<b><?php esc_html_e( 'Phone 1', 'iben-woo-elta-courier' ); ?></b>
						</label>

						<input type="text" id="iben_elta_order_metabox_phone1_single" class="iben_elta_order_metabox_phone1_single" name="iben_elta_order_metabox_phone1_single" value="" size="25" />
					</p>
					
					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_phone2_single">
							<b><?php esc_html_e( 'Phone 2', 'iben-woo-elta-courier' ); ?></b>
						</label>
						<input type="text" id="iben_elta_order_metabox_phone2_single" class="iben_elta_order_metabox_phone2_single" name="iben_elta_order_metabox_phone2_single" value="" size="25" />
					</p>
					
					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_comments_single">
							<b><?php esc_html_e( 'Comments', 'iben-woo-elta-courier' ); ?></b>
						</label>
						<textarea id="iben_elta_order_metabox_comments_single" class="iben_elta_order_metabox_comments_single" name="iben_elta_order_metabox_comments_single"></textarea>						
					</p>

					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_cash_delivery_single">
							<b><?php esc_html_e( 'Cash on delivery', 'iben-woo-elta-courier' ); ?></b>
						</label>

						<input type="text" id="iben_elta_order_metabox_cash_delivery_single" class="iben_elta_order_metabox_cash_delivery_single" name="iben_elta_order_metabox_cash_delivery_single" value="" size="25" />
					</p>	

					<p class="iben-elta-field">
						<label for="iben_elta_order_metabox_weight_single">
							<b><?php esc_html_e( 'Weight', 'iben-woo-elta-courier' ); ?></b>
						</label>
						<?php
						
						// check if default weight has been set and use this
						$default_weight = sanitize_text_field($main_settings['iben_elta_main_settings_default_weight']);

						if( isset( $default_weight ) && $default_weight > 0 ) {
							
							$weight = $default_weight;

						}else{						

							$weight = 0;
						}

						?>
						<input type="text" id="iben_elta_order_metabox_weight_single" class="iben_elta_order_metabox_weight_single" name="iben_elta_order_metabox_weight_single" value="<?php echo esc_attr( $weight ); ?>" size="25" />
					</p>				

					<div><a href='#' id ='iben_elta_order_create_voucher_more'><?php esc_html_e( 'More options', 'iben-woo-elta-courier' ); ?></a></div>

					<div class="iben_elta_order_loadmore_hidden_section">

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_packages_number_single">
								<b><?php esc_html_e( 'Number of parcels', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php $parcels = '1'; ?>
							<input type="text" id="iben_elta_order_metabox_packages_number_single" class="iben_elta_order_metabox_packages_number_single" name="iben_elta_order_metabox_packages_number_single" value="<?php echo esc_attr( $parcels ); ?>" size="25" />
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_ref_single">
								<b><?php esc_html_e( 'Reference No', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<input type="text" id="iben_elta_order_metabox_ref_single" class="iben_elta_order_metabox_ref_single" name="iben_elta_order_metabox_ref_single" value="" size="25" />
						</p>


						
						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_service_single">
								<b><?php esc_html_e( 'Service', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php
								$service = 1;						
							?>
							<select id="iben_elta_order_metabox_service_single" name="iben_elta_order_metabox_service_single">
								<option value="1" <?php echo ($service == '1' ? 'selected' : ''); ?> >1 - 201</option>
								<option value="2" <?php echo ($service == '2' ? 'selected' : ''); ?> >2 - 204 </option>
							</select>
						</p>

						<p class="iben-elta-field">
								<label for="iben_elta_order_metabox_subcode_single">
									<b><?php esc_html_e( 'Subcode', 'iben-woo-elta-courier' ); ?></b>
								</label>
								<?php
									$subcode = ''; 					
								?>
								<input type="text" id="iben_elta_order_metabox_subcode_single" class="iben_elta_order_metabox_subcode_single" name="iben_elta_order_metabox_subcode_single" value="<?php echo esc_attr( $subcode ); ?>" size="25" />
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_sur_1_single">
								<b><?php esc_html_e( 'Special management', 'iben-woo-elta-courier' ); ?></b>
							</label>
						
							<select id="iben_elta_order_metabox_sur_1_single" name="iben_elta_order_metabox_sur_1_single">
								<option value="0" ><?php esc_html_e( 'No', 'iben-woo-elta-courier' ); ?></option>
								<option value="1" ><?php esc_html_e( 'Yes', 'iben-woo-elta-courier' ); ?></option>
							</select>
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_sur_2_single">
								<b><?php esc_html_e( 'Delivery Time', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php
								$sur_2 = 0;
											
							?>							
							<select id="iben_elta_order_metabox_sur_2_single" name="iben_elta_order_metabox_sur_2_single">
								<option value="0" <?php echo ($sur_2 == 0 ? 'selected' : ''); ?>><?php esc_html_e( 'No', 'iben-woo-elta-courier' ); ?></option>
								<option value="1" <?php echo ($sur_2 == 1 ? 'selected' : ''); ?>><?php esc_html_e( 'Yes', 'iben-woo-elta-courier' ); ?></option>
							</select>
						</p>

						<p class="iben-elta-field">
							<label for="iben_elta_order_metabox_sur_3_single">
								<b><?php esc_html_e( 'Delivery Saturday', 'iben-woo-elta-courier' ); ?></b>
							</label>
							<?php
								$sur_3 = 0;
											
							?>							
							<select id="iben_elta_order_metabox_sur_3_single" name="iben_elta_order_metabox_sur_3_single">
								<option value="0" <?php echo ($sur_3 == 0 ? 'selected' : ''); ?>><?php esc_html_e( 'No', 'iben-woo-elta-courier' ); ?></option>
								<option value="1" <?php echo ($sur_3 == 1 ? 'selected' : ''); ?>><?php esc_html_e( 'Yes', 'iben-woo-elta-courier' ); ?></option>
							</select>
						</p>
					</div>			
					

					<p class="iben_elta-field-buttons iben-normal-shipping-button"> 

						<button type="button" id="iben_elta_order_create_single_voucher" class="button button-primary"><?php esc_html_e( 'Create voucher (SYDETA)', 'iben-woo-elta-courier' ); ?></button>
						
						<?php 
						if( isset( $main_settings['iben_elta_main_settings_2nd_enable_account'] ) ) {
							?>
							<span style="display: block; padding-top: 5px;">
								<select id="iben_elta_order_metabox_elta_account" name="iben_elta_order_metabox_elta_account">
									<option value="1"><?php echo apply_filters( 'iben_elta_filter_order_elta_1st_account_name', esc_html__( '1st account', 'iben-woo-elta-courier' )); ?></option>
									<option value="2"><?php echo apply_filters( 'iben_elta_filter_order_elta_2nd_account_name', esc_html__( '2nd account', 'iben-woo-elta-courier' )); ?></option>
								</select>
							</span>
						<?php } ?>

						<?php
						
						
						echo "<div class='iben_elta_settings_info_button'>";
						if( isset( $main_settings['iben_elta_main_settings_email_completed'] ) || isset( $main_settings['iben_elta_button_settings_create_send_completed_email'] ) || isset( $main_settings['iben_elta_button_settings_create_send_completed_email'] ) || isset( $main_settings['iben_elta_button_settings_create_pdf'] ) ) {
							echo '<span class="iben_elta_settings_info_button_title">'.esc_html__( 'According your settings, when you create the voucher:', 'iben-woo-elta-courier' ).'</span>';
						}
						echo "<ul>";

						if( isset( $main_settings['iben_elta_button_settings_create_pdf'] ) ) {
							$checkbox_create_pdf = '<input type="checkbox" class="iben_elta_create_voucher_settings" id="iben_elta_button_settings_create_pdf" name="iben_elta_main_settings_fields[iben_elta_button_settings_create_pdf]" value="1" ' . checked( 1, isset( $main_settings['iben_elta_button_settings_create_pdf'] ) ? $main_settings['iben_elta_button_settings_create_pdf'] : 0, false ) . '/>';
							echo '<li>'.esc_html__( 'PDF will be generated immediately', 'iben-woo-elta-courier' ). $checkbox_create_pdf .'</li>';
						}

						echo "</ul
						></div>";
						?>
					</p>





				</div>


				<div class="iben-elta-right">

					<div class="iben_elta_order_right_box_vouchers">
						<h3 class="iben_elta_order_box_title2"><?php esc_html_e( 'Vouchers', 'iben-woo-elta-courier' ); ?></h3>
						<div class='iben_elta_order_vouchers'>

							<span style="display: inline-block; padding: 5px 10px; vertical-align: top;">
								<span id="iben_elta_order_voucher_response" class="iben_elta_order_voucher_response"></span>
								<span style="display: block;" id="iben_elta_compine_vouchers_respond"> </span>
							</span>
						
							<span style="display: inline-block; padding: 5px 10px; vertical-align: top;">
								<?php	
									$regenerate = false;
								?>
									<span id="iben_elta_order_returned_vouchers_response" class="iben_elta_order_returned_vouchers_response">
								</span>
								<span style="display: block;" id="iben_elta_compine_returned_vouchers_respond"> </span>
								<span style="display: inline-flex;align-content: center;align-items: center;">
									<button  type="button" class="button iben_elta_order_button_compine_ret_vouchers button-secondary"  style="display:none;"><?php ($regenerate ? esc_html_e( 'Re generate PDF', 'iben-woo-elta-courier' ) : esc_html_e( 'Generate and print PDF', 'iben-woo-elta-courier' )) ?></button>
									<div class="iben_elta_compine_returned_vouchers_spinner iben_elta_spinners" style="display:none;"><div class="bounce1"></div><div class="bounce2"></div><div class="bounce3"></div></div>
								</span>
							</span>
						
						</div>
					</div>
						
					<div class="iben_elta_order_right_box_tools">
						<h3 class="iben_elta_order_box_title2"><?php esc_html_e( 'Tools', 'iben-woo-elta-courier' ); ?></h3>
						<div style="display: block;">

							<div style="display: block; padding: 5px 0px; text-align: left;"> 
								<select style="width: 100px;" id="iben_elta_order_tracing_voucher_ref_single" name="iben_elta_order_tracing_voucher_ref_single">
									<option value="1" ><?php esc_html_e( 'By Voucher', 'iben-woo-elta-courier' ); ?></option>
									<option value="2" ><?php esc_html_e( 'By ref', 'iben-woo-elta-courier' ); ?></option>
								</select>
								<input type="text" style="width: 100px;" id="iben_elta_order_tracing_voucher_single" value="">
								<button style="" type="button" id="iben_elta_order_tracing_button_single" class="button iben_elta_order_tracing_button_single button-secondary"><?php esc_html_e( 'Tracking voucher', 'iben-woo-elta-courier' ); ?></button>
							</div>
							
							<div style="display: block; padding: 5px 0px; text-align: left;"> 
								<input type="text" id="iben_elta_order_station_voucher_single" placeholder = '<?php esc_html_e( 'Search ELTA Courier with ZIP', 'iben-woo-elta-courier' ); ?>' value="">
								<button style="" type="button" id="iben_elta_order_station_button_single" class="button iben_elta_order_station_button_single button-secondary"><?php esc_html_e( 'Check Station', 'iben-woo-elta-courier' ); ?></button>
							</div>
						</div>

						<span id="iben_elta_order_tools_response_single" class='iben_elta_order_box_tools_inner'></span>							
					</div>
					<?php
					do_action('iben_elta_order_right_single');
					?>

				</div>		
			</div>

			<?php
			return ob_get_clean();
		}		


		/**
		 *
		 * @since     1.0.0
		 */
		public function iben_elta_add_order_metabox(){

			$order = null;
			if (!$order && isset($_GET['id'])) {
				$order_id = intval($_GET['id']);
				$order = wc_get_order($order_id);
			}elseif (!$order && isset($_GET['post'])) {
				$order_id = intval($_GET['post']);
				$order = wc_get_order($order_id);
			}

			if ( !is_a( $order, 'WC_Order' )  ){
				return;
			}

			$billing_country = $order->get_billing_country();
			$shipping_country = $order->get_shipping_country();			

			if ( $billing_country != $shipping_country && $shipping_country != '' ){
				$country = $shipping_country;
			}else{
				$country = $billing_country;
			}

			$order_status = $order->get_status();
			
			$list_countries = array_keys($this->countries);
			$list_countries = apply_filters('iben_elta_courier_order_metabox_display_countries', $list_countries);

			$except_list_order_states =  array("pending", "cancelled", "failed" );
			$except_list_order_states = apply_filters('iben_elta_courier_order_metabox_not_display_order_status', $except_list_order_states);			

			$get_vouchers = iben_elta_get_Vouchers_from_id_order($order->get_id());
			

			if (($country == '' || in_array($country, $list_countries)) && !in_array($order_status, $except_list_order_states) && apply_filters('iben_elta_filter_display_create_voucher_box', true, $order) || !empty($get_vouchers) ) {
				
				$screen_id = 'shop_order';
			
				if (defined('WC_VERSION') && version_compare(WC_VERSION, '7.7.0', '>=')) {
					if (class_exists('Automattic\WooCommerce\Utilities\OrderUtil')) {
						$orderUtilEnabled = Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
						
						if ($orderUtilEnabled) {
							$screen_id = wc_get_container()->get(CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled()
								? wc_get_page_screen_id('shop-order')
								: 'shop_order';
						}
					}
				}
			
				add_meta_box(
					'iben_elta_courier_order_metabox',
					'Create ELTA voucher',
					array( $this , 'iben_elta_meta_box'),
					$screen_id,
					'normal',
					'high'
				);
			}			
		
		}

		/**
		 *
		 * @since     1.0.0
		 */		
		
		public function iben_elta_meta_box($post_or_order_object){
			$order = ( $post_or_order_object instanceof WP_Post ) ? wc_get_order( $post_or_order_object->ID ) : $post_or_order_object;			
			
			if ( !is_a( $order, 'WC_Order' )  ){
				return;
			}
			echo $this->iben_elta_meta_box_html($order );
		}		

		

		/**
		 *
		 * @since     2.8.0
		 * 
		 */
		private function iben_elta_meta_box_sms_single(){
			// wp_nonce_field( 'iben_elta_send_sms_single_action', 'iben_elta_send_sms_single' );
			$message = '';
			$options = get_option( 'iben_elta_main_settings_fields' );
			if ( isset($options['iben_elta_main_settings_single_order_sms_enable_default_message']) && $options['iben_elta_main_settings_single_order_sms_enable_default_message'] != '' ){
				$message = $options['iben_elta_main_settings_single_order_sms_enable_default_message'];
			}
			?>

			<h3 class="iben_elta_order_box_singe_sms">Send SMS</h3>

			<p class="form-field form-field-wide" style="margin: 0px;padding: 0px;">
				<textarea id="iben_elta_send_sms_single_message" class="iben_elta_send_sms_single_message" name="iben_elta_send_sms_single_message" rows="5" cols="30"><?php echo $message; ?></textarea>
			</p>
			<p class="form-field form-field-wide" style="margin: 0px;padding: 0px;">
				<input type="text" id="iben_elta_send_sms_single_mobile" class="iben_elta_send_sms_single_mobile" name="iben_elta_send_sms_single_mobile" placeholder=<?php esc_html_e( 'Mobile phone', 'iben-woo-elta-courier' ); ?> value="" size="25" />
			</p>
			<p class="form-field form-field-wide" style="margin: 0px;padding: 0px;">
				<input type="text" id="iben_elta_send_sms_single_voucher_number" class="iben_elta_send_sms_single_voucher_number" placeholder=<?php esc_html_e( 'Voucher number', 'iben-woo-elta-courier' ); ?> name="iben_elta_send_sms_single_voucher_number" value="" size="25" />
			</p>
			<p class="form-field form-field-wide" style="margin: 0px;padding: 5px:0px;"> 
				<span id="iben_elta_send_sms_single_response" style="color: red; font-weight: bold;"></span>
				<button type="button" id="iben_elta_send_sms_single" class="button iben_sendsmslexicon button-primary" name="iben_sendsms"><?php esc_html_e( 'Send SMS', 'iben-woo-elta-courier' ) ?></button>
			</p>
			<p class="form-field form-field-wide" style="margin: 0px;padding: 0px;">
				<a href="#" class="iben_sms_lexicon_check_balance"><?php esc_html_e( 'Check balance', 'iben-woo-elta-courier' ); ?></a>
			</p> 
			<?php
		}
		/**
		 *
		 * @since     2.8.0
		 * 
		 */
		public function iben_elta_sms_v2(){
			echo $this->iben_elta_meta_box_sms_single();
			
		}



		/**
		 *
		 * @since     1.0.0
		 * 
		 */
		public function iben_elta_create_voucher_handler(){
		
			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				

				$order = wc_get_order( absint( $data['order_id'] ) );

				if ( !is_a( $order, 'WC_Order' ) ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Order does not exist', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();	
				}	
				
				$order_id  = $order->get_id();

				$confirm_duplicated_voucher = absint($data['confirm_duplicated_voucher']);
				
				$qry = iben_elta_get_Vouchers_from_id_order( $order_id );
				
				if( $qry && $confirm_duplicated_voucher == 0 ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Voucher already exist for this order', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();					
				}

				//iben *Geniki compatibility
				if ( class_exists( 'Iben_Woo_Geniki' ) && function_exists('iben_geniki_get_Vouchers_from_id_order') ) {

					$qrys = iben_geniki_get_Vouchers_from_id_order($order_id);

					if( $qrys && $confirm_duplicated_voucher == 0 ){

						$resultss = array (
							"status" => "failure",
							"message" => 'ERROR',	
							"result" => esc_html__( 'Geniki Voucher already exist for this order', 'iben-woo-elta-courier' )
						);
						echo json_encode ( $resultss );
						wp_die();					
					}
				}

				//iben *ACS compatibility
				if ( class_exists( 'Iben_Woo_Acs' ) && function_exists('iben_acs_get_Vouchers_from_id_order') ) {

					$qrys = iben_acs_get_Vouchers_from_id_order($order_id);

					if( $qrys && $confirm_duplicated_voucher == 0 ){

						$resultss = array (
							"status" => "failure",
							"message" => 'ERROR',	
							"result" => esc_html__( 'ACS Voucher already exist for this order', 'iben-woo-elta-courier' )
						);
						echo json_encode ( $resultss );
						wp_die();					
					}
				}

				//iben *Speedex compatibility
				if ( class_exists( 'Iben_Woo_Speedex' ) && function_exists('iben_speedex_get_Vouchers_from_id_order') ) {

					$qrys = iben_speedex_get_Vouchers_from_id_order($order_id);

					if( $qrys && $confirm_duplicated_voucher == 0 ){

						$resultss = array (
							"status" => "failure",
							"message" => 'ERROR',	
							"result" => esc_html__( 'Speedex Voucher already exist for this order', 'iben-woo-elta-courier' )
						);
						echo json_encode ( $resultss );
						wp_die();					
					}
				}									

				$types = array();

				$types['fullname'] = sanitize_text_field($data['fullname']);
				$types['address'] = sanitize_text_field($data['address']);
				$types['region'] = sanitize_text_field($data['region']);
				
				$types['postal_code'] = sanitize_text_field($data['postal_code']); //sanitize_html_class
				$types['phone1'] = sanitize_text_field($data['phone1']);
				$types['phone2'] = sanitize_text_field($data['phone2']);
				$types['service'] = sanitize_text_field($data['service']);
				$types['weight'] = sanitize_text_field($data['weight']);
				$types['packages_number'] = sanitize_text_field($data['packages_number']);
				$types['subcode'] = sanitize_text_field($data['subcode']);

				$types['comments'] = sanitize_text_field($data['comments']);
				$types['cash_delivery'] = ( sanitize_text_field($data['cash_delivery']) == '' ||  sanitize_text_field($data['cash_delivery']) < 0 || sanitize_text_field($data['cash_delivery']) == 'NaN' ? '0' : sanitize_text_field($data['cash_delivery']));
				
				$types['sur_1'] = '0';
				$types['sur_2'] = sanitize_text_field($data['sur_2']);
				$types['sur_3'] = sanitize_text_field($data['sur_3']);
				
				$types['ref'] = sanitize_text_field($data['ref']);
				$types['second_account'] = sanitize_text_field($data['second_account']);
				
				$types = apply_filters( 'iben_elta_filter_types_order_metabox', $types, $order_id );

				$webservice_run = $this->ibenEltaWeServiceCreateVoucher($types);


				if ($webservice_run['state'] === false){
					$message = esc_html__('Voucher didn\'t created!', 'iben-woo-elta-courier');
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $webservice_run['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				if ($webservice_run['state'] === true){
					
					$webservice_run = $webservice_run['result'];
				
					$getVg_child = array();

					array_push($getVg_child, $webservice_run->getVg_code());
					$main_voucher = $webservice_run->getVg_code();

					$child_vouchers = array_filter($webservice_run->getVg_child());

					if( !empty($child_vouchers) && sizeof($child_vouchers) != 0) {

						foreach ($child_vouchers as $voucher){
							array_push($getVg_child, $voucher);
						}
					
					}

					$current_user = wp_get_current_user();

					$default = array(
						'logged_user' => esc_html( $current_user->user_login),
						'id_user' => get_current_user_id(),
						'id_order' => $order_id,
						'id_elta_reason' => 1,
						'elta_returned_vouchers' => '',
						'elta_delivery_vouchers' => json_encode($getVg_child),
						'elta_pdf_type' => $this->iben_elta_main_settings_papersize,
						'elta_voucher_info' => maybe_serialize($types),
						'id_elta_account' => $types['second_account'],
						'duplicated_record' => ( $confirm_duplicated_voucher == 1 ? 1 : 0 ),
						'time' => current_time('timestamp'),
						'plugin_version' => $this->version
					);


					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
					$result = $wpdb->insert($table_name, $default);

					do_action( 'iben_elta_hook_voucher_created', $order_id, $main_voucher );

					$order->update_meta_data( '_iben_elta_voucher_data', $types);	
					$order->update_meta_data( '_iben_elta_voucher_ship_number', $main_voucher);	

		

					$export = '';
					$note = '';
					$get_vouchers = iben_elta_get_Vouchers_from_id_order($order_id);

					foreach ($get_vouchers as $voucher){
						$export .= $this->iben_display_html_vouchers($voucher->time, $voucher->elta_delivery_vouchers, $order_id);
						$note .= $this->iben_display_html_vouchers($voucher->time, $voucher->elta_delivery_vouchers, $order_id, true);
					}

					if ( !empty($getVg_child) && $getVg_child != ''){

						$order->update_meta_data( '_iben_elta_voucher_generated_vouchers', json_encode($getVg_child) );
					}


					$position_page = sanitize_text_field($data['position_page']);
					
					$exportsettings = array();
					$exportsettings['set_order_completed'] = 0;
					$exportsettings['send_completed_email'] = 0;
					$exportsettings['create_pdf'] = 0;
					$exportsettings['tracking'] = $main_voucher;
					
					$main_settings = $this->main_settings;
					
					// always before SEND the email. 
					// if setting. Attach tracking info is enabled
					if( isset( $main_settings['iben_elta_main_settings_email_completed'] ) && absint( $data['attach_trackinginfo_in_completed']) == 1 ) {

						$order->update_meta_data( 'iben_elta_tracking_order_notification', 1 );

					}else if( isset( $main_settings['iben_elta_main_settings_email_completed'] ) && absint( $data['attach_trackinginfo_in_completed']) == 0 ) {
						
						$order->update_meta_data( 'iben_elta_tracking_order_notification', 0 );
					}

					//if setting. Set order completed, is enabled
					if( isset( $main_settings['iben_elta_main_settings_voucher_change_order_status'] ) && $main_settings['iben_elta_main_settings_voucher_change_order_status'] != -1) {

						$status_value = iben_elta_get_order_status_value($main_settings['iben_elta_main_settings_voucher_change_order_status'] );

						$order->update_status( $status_value );
						$exportsettings['set_order_completed'] = 1;

					}

					//if setting. Send completed order is enabled
					if( isset( $main_settings['iben_elta_button_settings_create_send_completed_email'] ) && absint( $data['send_completed_email']) == 1) {

						$mailer = WC()->mailer()->get_emails();
						$mailer['WC_Email_Customer_Completed_Order']->trigger( $order_id );
						$exportsettings['send_completed_email'] = 1;

					}
					$order->add_order_note( sprintf(esc_html__( 'Voucher created: %s', 'iben-woo-elta-courier' ), $note ) );
					$order->save();

					$result1 = $export;

					$errorpdf = '';
					//if setting. Create PDF is enabled
					if( isset( $main_settings['iben_elta_button_settings_create_pdf'] ) && absint( $data['create_pdf']) == 1 ) {

						$get_vouchers = iben_elta_get_Array_Vouchers_from_id_order($order_id);

						if( !empty($get_vouchers) || sizeof( $get_vouchers ) != 0 ){
			
							$count_vouchers = count($get_vouchers['elta_delivery_vouchers']) ;
		
							require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
							$pdf = new \Clegginabox\PDFMerger\PDFMerger;
							$output = '';
		
							foreach ($get_vouchers['elta_delivery_vouchers'] as $item) {
		
								if ($item != ''){
									$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($item), $types['second_account']);
		
									if ( $pdf_created['state'] !== true){
										
										$errorpdf.= '('.$item.') <br/>'.esc_html__('This PDF didnt created. You have to Regenerate it!', 'iben-woo-elta-courier').' <span class="iben_elta_pdf_error">'.$pdf_created['result'].'</span>';
		
									}else{
										
										$pdf->addPDF($pdf_created['result'], 'all');
		
									}
								}
							}//foreach
		
		
							$output.= $this->iben_display_html_vouchers('time', json_encode($get_vouchers['elta_delivery_vouchers']), $order_id );
		
							if ($count_vouchers > 1 && $errorpdf == '' ){
		
								$random_string = $this->iben_generateRandomString();
								$name_pdf = $order_id.$random_string;

								$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';

								$pdf->merge('file', $merge, 'P');
								$output .= iben_elta_Compined_Voucher_Link($name_pdf);
		
								global $wpdb;
								$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
					
								$wpdb->update( 
									$table_name, 
									array( 
										'elta_delivery_compined_vouchers' => $name_pdf
									), 
									array( 'id_order' => $order_id, 'id_elta_reason' => 1 )
								);
								
								$default2 = array(
									'logged_user' => esc_html( $current_user->user_login),
									'id_user' => get_current_user_id(),
									'id_arrays' => json_encode($order_id),
									'voucher_arrays' => json_encode($get_vouchers['elta_delivery_vouchers']),
									'compined_pdf' => $name_pdf,
									'time' => current_time('timestamp')
								);
								$table_name2 = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
								$result = $wpdb->insert($table_name2, $default2);

							}

		
						}//if $get_vouchers exists
							
						$exportsettings['create_pdf'] = 1;
						
						$result1 = $output;
						
						if (isset($errorpdf)){
							$result1 = $output.'<br/>'.$errorpdf;
						}
						
					}

					$message = esc_html__('Voucher generated!', 'iben-woo-elta-courier');

					$resultss = array (
						"status" => "success",
						"message" => $message,
						"exportsettings" => $exportsettings,
						"result" => $result1
					);
			
					echo json_encode ( $resultss );
					
				}		

		
			}
		
			// ajax handlers must die
			wp_die();
		}

		/**
		 *
		 * @since     1.0.0
		 * 
		 */
		public function iben_elta_create_returned_voucher_ajax_handler(){
		
			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  
				
				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$order = wc_get_order( absint( $data['order_id'] ) );

				if ( !is_a( $order, 'WC_Order') ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Order does not exist', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();	
				}		  
				
				$order_id  = $order->get_id();
				$confirm_duplicated_returned_voucher = absint($data['confirm_duplicated_returned_voucher']);
			
				$qry = iben_elta_get_ReturnedVouchers_from_id_order( $order_id  );

				if( $qry && $confirm_duplicated_returned_voucher == 0 ){

					$resultss = array (
						"status" => "failure",
						"result" => "Returned voucher already exist"
					);
					echo json_encode ( $resultss );
					wp_die();					
				}

				$types = array();
				$types['fullname'] = sanitize_text_field($data['fullname']);
				$types['address'] = sanitize_text_field($data['address']);
				$types['region'] = sanitize_text_field($data['region']);
				
				$types['postal_code'] = sanitize_text_field($data['postal_code']); //sanitize_html_class
				$types['phone1'] = sanitize_text_field($data['phone1']);
				$types['phone2'] = sanitize_text_field($data['phone2']);
				$types['service'] = sanitize_text_field($data['service']);
				$types['weight'] = sanitize_text_field($data['weight']);
				$types['packages_number'] = sanitize_text_field($data['packages_number']);
				$types['subcode'] = sanitize_text_field($data['subcode']);

				$types['comments'] = sanitize_text_field($data['comments']);
				$types['cash_delivery'] = ( sanitize_text_field($data['cash_delivery']) == '' ||  sanitize_text_field($data['cash_delivery']) < 0 || sanitize_text_field($data['cash_delivery']) == 'NaN' ? '' : sanitize_text_field($data['cash_delivery']));

				//important field to enable return
				$types['sur_1'] = '1';
				$types['sur_2'] = sanitize_text_field($data['sur_2']);
				$types['sur_3'] = sanitize_text_field($data['sur_3']);
				$types['ref'] = sanitize_text_field($data['ref']);
				$types['second_account'] = sanitize_text_field($data['second_account']);
		
				$types = apply_filters( 'iben_elta_filter_returned_types_order_metabox', $types, $order_id );
				$webservice_run = $this->ibenEltaWeServiceCreateVoucher($types);
				

				if ($webservice_run['state'] === false){
					$message = esc_html__('Voucher didn\'t created!', 'iben-woo-elta-courier');
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $webservice_run['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}



				if ($webservice_run['state'] === true){
					
					$webservice_run = $webservice_run['result'];

					$getVg_child = array();

					array_push($getVg_child, $webservice_run->getReturn_vg());
					array_push($getVg_child, $webservice_run->getVg_code());
					
					$child_vouchers = array_filter($webservice_run->getVg_child());
										
					if( !empty($child_vouchers) && sizeof($child_vouchers) != 0) {

						foreach ($child_vouchers as $voucher){
							array_push($getVg_child, $voucher);
						}
					
					}
										
					
					$current_user = wp_get_current_user();

					$default = array(						
						'logged_user' => esc_html( $current_user->user_login),
						'id_user' => get_current_user_id(),
						'id_order' => $order_id ,
						'id_elta_reason' => 2,
						'duplicated_record' => ( $confirm_duplicated_returned_voucher == 1 ? 1 : 0 ),
						'elta_delivery_vouchers' => '',
						'elta_returned_vouchers' => json_encode($getVg_child),
						'elta_pdf_type' => $this->iben_elta_main_settings_papersize,
						'elta_voucher_info' => maybe_serialize($types),
						'id_elta_account' => $types['second_account'],
						'time' => current_time('timestamp'),
						'plugin_version' => $this->version,						
					);


					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
					$result = $wpdb->insert($table_name, $default);

					$order->update_meta_data( '_iben_elta_returned_vouchers_generated', 1 );
					$order->update_meta_data( '_iben_elta_returned_voucher_data', $types );
					$order->save();
					
					$export = '';
					$note = '';

					$get_vouchers = iben_elta_get_ReturnedVouchers_from_id_order($order_id );

					foreach ($get_vouchers as $voucher){
						
						$export .= $this->iben_display_html_returned_vouchers($voucher->time, $voucher->elta_returned_vouchers, $order_id );
						$note .= $this->iben_display_html_returned_vouchers($voucher->time, $voucher->elta_returned_vouchers, $order_id , true);

					}


					$order->add_order_note( sprintf(esc_html__( 'Voucher created: %s', 'iben-woo-elta-courier' ), $note ) );
					
					$result1 = $export;

					$message = esc_html__('Voucher created!', 'iben-woo-elta-courier');
		
					$resultss = array (
						"status" => "success",
						"message" => $message,
						"result" => $result1
					);
			
					echo json_encode ( $resultss );
					
				}
			
		
			}
		
			// ajax handlers must die
			wp_die();
		}	


		/**
		 *
		 * @since     1.3.0
		 */
		public function iben_elta_compine_single_vouchers_ajax_handler(){
			
			if (isset($_POST['iben_data'])) {
				
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];			

				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				$id = absint( $data['id'] );
				$id_elta_reason = absint( $data['id_elta_reason'] );

				if ($id_elta_reason == 1){
					$get_vouchers = iben_elta_get_Array_Vouchers_from_id($id);
				}else{
					$get_vouchers = iben_elta_get_Array_Returned_Vouchers_from_id($id);
				}

				global $wpdb;
				$table_name = $wpdb->prefix . 'iben_elta_vouchers';
		
				$result = $wpdb->get_row($wpdb->prepare("SELECT id_elta_account FROM $table_name WHERE id = %d", $id));  

				if ($result){

					$elta_account = $result->id_elta_account;

				}


				if( empty($get_vouchers) || sizeof( $get_vouchers ) == 0 ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Vouchers doesnt exist on DB', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();

				}else{

					$count_vouchers = count($get_vouchers) ;

					require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
					$pdf = new \Clegginabox\PDFMerger\PDFMerger;

					$output = '';
					$errorpdf = '';
					foreach ($get_vouchers as $item) {

						if ($item != ''){

							$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($item), $elta_account);

							if ( $pdf_created['state'] !== true){
								
								$errorpdf.= '('.$item.') <br/>'.esc_html__('This PDF didnt created. You have to Regenerate it!', 'iben-woo-elta-courier').' <span class="iben_elta_pdf_error">'.$pdf_created['result'].'</span>';
								
							}else{
								
								$pdf->addPDF($pdf_created['result'], 'all');
								
							}
						}
					}//foreach


					$output.= $this->iben_display_html_vouchers('time', json_encode($get_vouchers), $id );
					

					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';	

					if ($count_vouchers > 1 && $errorpdf == '' ){
						$random_string = $this->iben_generateRandomString();
						$name_pdf = $random_string.'_SV';
						$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
						
						$pdf->merge('file', $merge, 'P');						

						$output .= iben_elta_Compined_Voucher_Link($name_pdf);
						
						// if ειδικη διαχειριση
						if ($id_elta_reason == 1){

							$wpdb->update( 
								$table_name, 
								array( 
									'elta_delivery_compined_vouchers' => $name_pdf
								), 
								array( 'id' => $id)
							);

						}else{
							$wpdb->update( 
								$table_name, 
								array( 
									'elta_returned_compined_vouchers' => $name_pdf,
									'pdf_deleted' => 0
								), 
								array( 'id' => $id)
							);							

						}
					}

					if ($count_vouchers > 0){
						$wpdb->update( 
							$table_name, 
							array( 
								'pdf_deleted' => 0
							), 
							array( 'id' => $id)
						);	
					}
					$result = $output;
					
					if (isset($errorpdf)){
						$result = $output.'<br/>'.$errorpdf;
					}

					$message = 'Generated';

					$resultss2 = array (
						"status" => "success",
						"message" => $message,
						"result" => $result
					);
			
					echo json_encode ( $resultss2 );


				}//if $get_vouchers exists

			
			}

			wp_die();
		}

		/**
		 * Generate a random string
		 *
		 * @since     1.3.0
		 */
		function iben_generateRandomString($length = 10) {
			$characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
			$randomString = '';
			for ($i = 0; $i < $length; $i++) {
				$randomString .= $characters[wp_rand(0, strlen($characters) - 1)];
			}
			return $randomString;
		}

		/**
		 *
		 * @since     1.3.0
		 */
		public function iben_elta_create_single_voucher_handler(){
		
			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				
				$order_id  = -1;

				$types = array();

				$types['fullname'] = sanitize_text_field($data['fullname']);
				$types['address'] = sanitize_text_field($data['address']);
				$types['region'] = sanitize_text_field($data['region']);
				
				$types['postal_code'] = sanitize_text_field($data['postal_code']); //sanitize_html_class
				$types['phone1'] = sanitize_text_field($data['phone1']);
				$types['phone2'] = sanitize_text_field($data['phone2']);
				$types['service'] = sanitize_text_field($data['service']);
				$types['weight'] = sanitize_text_field($data['weight']);
				$types['packages_number'] = sanitize_text_field($data['packages_number']);
				$types['subcode'] = sanitize_text_field($data['subcode']);

				$types['comments'] = sanitize_text_field($data['comments']);
				$types['cash_delivery'] = ( sanitize_text_field($data['cash_delivery']) == '' ||  sanitize_text_field($data['cash_delivery']) < 0 || sanitize_text_field($data['cash_delivery']) == 'NaN' ? '' : sanitize_text_field($data['cash_delivery']));
				
				$types['sur_1'] = sanitize_text_field($data['sur_1']);
				$types['sur_2'] = sanitize_text_field($data['sur_2']);
				$types['sur_3'] = sanitize_text_field($data['sur_3']);
				$types['ref'] = sanitize_text_field($data['ref']);
				$types['second_account'] = sanitize_text_field($data['second_account']);

				$webservice_run = $this->ibenEltaWeServiceCreateVoucher($types);

				if ($webservice_run['state'] === false){
					$message = esc_html__('Voucher didn\'t created!', 'iben-woo-elta-courier');
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $webservice_run['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				if ($webservice_run['state'] === true){
					
					$webservice_run = $webservice_run['result'];
				
					$getVg_child = array();

					if ( $types['sur_1'] == 1 && $webservice_run->getReturn_vg() ){
						array_push($getVg_child, $webservice_run->getReturn_vg());
					}
					array_push($getVg_child, $webservice_run->getVg_code());
					$main_voucher = $webservice_run->getVg_code();

					$child_vouchers = array_filter($webservice_run->getVg_child());

					if( !empty($child_vouchers) && sizeof($child_vouchers) != 0) {

						foreach ($child_vouchers as $voucher){
							array_push($getVg_child, $voucher);
						}
					
					}


					$current_user = wp_get_current_user();
					

					// Ειδικη διαχειριση ή οχι
					//ναι
					if ($types['sur_1'] == 1){

						$id_elta_reason = 2;
						$elta_delivery_vouchers = '';
						$elta_returned_vouchers = json_encode($getVg_child);

					}else{

						$id_elta_reason = 1;
						$elta_delivery_vouchers = json_encode($getVg_child);
						$elta_returned_vouchers = '';

					}

					$default = array(
						'logged_user' => esc_html( $current_user->user_login),
						'id_user' => get_current_user_id(),
						'id_order' => $order_id,
						'id_elta_reason' => $id_elta_reason,
						'elta_delivery_vouchers' => $elta_delivery_vouchers,
						'elta_returned_vouchers' => $elta_returned_vouchers,
						'elta_pdf_type' => $this->iben_elta_main_settings_papersize,
						'elta_voucher_info' => maybe_serialize($types),
						'duplicated_record' => 0,
						'id_elta_account' => $types['second_account'],
						'time' => current_time('timestamp'),
						'plugin_version' => $this->version
					);


					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
					$result = $wpdb->insert($table_name, $default);

					$lastid = $wpdb->insert_id;

					$export = '';

					foreach ($getVg_child as $voucher){
						
						$export .= $voucher.'<br/>';

					}
					$export .= 'Δημιουργήστε τα PDF από την λίστα';
					
					$exportsettings = array();
					$exportsettings['create_pdf'] = 0;

					$main_settings = $this->main_settings;
					
					//if setting. Create PDF is enabled
					if( isset( $main_settings['iben_elta_button_settings_create_pdf'] ) && absint( $data['create_pdf']) == 1 ) {

						$get_vouchers = $getVg_child;
						$errorpdf = '';
						if( !empty($get_vouchers) || sizeof( $get_vouchers ) != 0 ){
			
							$count_vouchers = count($get_vouchers) ;
		
							require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
							$pdf = new \Clegginabox\PDFMerger\PDFMerger;
							$output = '';
		
							foreach ($get_vouchers as $item) {
		
								if ($item != ''){
									$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($item), $types['second_account']);
		
									if ( $pdf_created['state'] !== true){
		
										$errorpdf.= '('.$item.') <br/>'.esc_html__('This PDF didnt created. You have to Regenerate it!', 'iben-woo-elta-courier').' <span class="iben_elta_pdf_error">'.$pdf_created['result'].'</span>';

									}else{
										
										$pdf->addPDF($pdf_created['result'], 'all');
		
									}
								}
							}//foreach
		

							$output.= $this->iben_display_html_vouchers('time', json_encode($get_vouchers), $order_id );
		
							if ($count_vouchers > 1 && $errorpdf == '' ){
								$random_string = $this->iben_generateRandomString();
								$name_pdf = $random_string.'_SV';

								$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
								$pdf->merge('file', $merge, 'P');
		
								$output .= iben_elta_Compined_Voucher_Link($name_pdf);
		
								global $wpdb;
								$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
								
								// nai
								if ($types['sur_1'] == 1){
									$wpdb->update( 
										$table_name, 
										array( 
											'elta_returned_compined_vouchers' => $name_pdf
										), 
										array( 'id' => $lastid )
									);
								}else{
									$wpdb->update( 
										$table_name, 
										array( 
											'elta_delivery_compined_vouchers' => $name_pdf
										), 
										array( 'id' => $lastid )
									);									

								}
							}
		
						}//if $get_vouchers exists
							
						$exportsettings['create_pdf'] = 1;
						
						$export = $output;
						
						
						if (isset($errorpdf)){
							$export = $output.'<br/>'.$errorpdf;
						}

					}
					

					if (iben_elta_is_greek_mobile($types['phone1'])){
						$mobile_number = $types['phone1'];
					}elseif (iben_elta_is_greek_mobile($types['phone2'])){
						$mobile_number = $types['phone2'];
					}else{
						$mobile_number = '';
					}
		

					$message = esc_html__('Voucher generated!', 'iben-woo-elta-courier');

					$resultss = array (
						"status" => "success",
						"message" => $message,
						"exportsettings" => $exportsettings,
						"main_voucher" => $main_voucher,
						"mobile_number" => $mobile_number,
						"result" => $export
					);
			
					echo json_encode ( $resultss );
					
				}
		
			}
		
			// ajax handlers must die
			wp_die();
		}
		
		/**
		 *
		 * @since     1.3.0
		 */
		public function iben_elta_save_single_contact_handler(){
		
			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				
				
				$error = false;
				$errormsg = '';

				$id_referrence = sanitize_text_field($data['id_referrence']);
				$contact_name = sanitize_text_field($data['fullname']);
				$contact_address = sanitize_text_field($data['address']);
				$contact_region = sanitize_text_field($data['region']);				
				$contact_postal = sanitize_text_field($data['postal_code']);
				$contact_tel1 = sanitize_text_field($data['phone1']);
				$contact_tel2 = sanitize_text_field($data['phone2']);
				$contact_comment = sanitize_text_field($data['comments']);


				if ($id_referrence == '' ){
					$error = true;
					$errormsg .= esc_html__('Code is empty', 'iben-woo-elta-courier');
					$errormsg .= '<br/>';
				}
				
				if ($contact_name == '' ){
					$error = true;
					$errormsg .= esc_html__('Contact name is empty', 'iben-woo-elta-courier');
					$errormsg .= '<br/>';
				}
				
				if ($contact_tel1 == '' && $contact_tel2 == ''){
					$error = true;
					$errormsg .= esc_html__('One telephone is required', 'iben-woo-elta-courier');
					$errormsg .= '<br/>';
				}	

				if ($contact_address == '' ){
					$error = true;
					$errormsg .= esc_html__('Contact address is empty', 'iben-woo-elta-courier');
					$errormsg .= '<br/>';
				}

				if ($error){

					$resultss = array (
						"status" => "failure",
						"result" => $errormsg,
					);
					echo json_encode ( $resultss );
					wp_die();

				}



				global $wpdb;
				$table_name = $wpdb->prefix . apply_filters('iben_elta_contacts_table_name', 'iben_elta_contacts');	


				$cntSQL = "SELECT count(*) as count FROM {$table_name} where id_reference='".$id_referrence."' AND id_reference != '-1' ";
				$record = $wpdb->get_results($cntSQL, OBJECT);
	
				if($record[0]->count == 0){

					$default = array(						
						'id_reference' => $id_referrence,
						'id_user' => get_current_user_id(),
						'contact_name' => $contact_name,
						'contact_address' => $contact_address,
						'contact_region' => $contact_region,
						'contact_postal' => $contact_postal,
						'contact_tel1' => $contact_tel1,
						'contact_tel2' => $contact_tel2,
						'contact_comment' => $contact_comment,
						'time_updated' => 0,
						'time_created' => current_time('timestamp')
					);
	
					$result = $wpdb->insert($table_name, $default);	
					$lastid = $wpdb->insert_id;

					$result1 = esc_html__('Contact Saved!', 'iben-woo-elta-courier');

				}else{


					$wpdb->update( 
						$table_name, 
						array( 
							'id_user' => get_current_user_id(),							
							'contact_name' => $contact_name,
							'contact_address' => $contact_address,
							'contact_region' => $contact_region,
							'contact_postal' => $contact_postal,
							'contact_tel1' => $contact_tel1,
							'contact_tel2' => $contact_tel2,
							'contact_comment' => $contact_comment,
							'time_updated' => current_time('timestamp')
						), 
						array( 'id_reference' => $id_referrence )
					);

					$result1 = esc_html__('Contact Updated!', 'iben-woo-elta-courier');

				}
				

				$resultss = array (
					"status" => "success",
					"result" => $result1
				);
		
				echo json_encode ( $resultss );
		
			}
		
			// ajax handlers must die
			wp_die();
		}


		/**
		 *
		 * @since     1.3.0
		 */
		public function iben_elta_load_single_contact_handler(){
		
			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
					
				
				$id_referrence = sanitize_text_field($data['id_referrence']);
				$exportsettings = array();


				global $wpdb;
				$table_name = $wpdb->prefix . apply_filters('iben_elta_contacts_table_name', 'iben_elta_contacts');
		 
				$result = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id_reference = %s", $id_referrence));  
		
				if ($result){

					$exportsettings['contact_name'] = $result->contact_name;
					$exportsettings['contact_address'] = $result->contact_address;
					$exportsettings['contact_region'] = $result->contact_region;
					$exportsettings['contact_postal'] = $result->contact_postal;
					$exportsettings['contact_tel1'] = $result->contact_tel1;
					$exportsettings['contact_tel2'] = $result->contact_tel2;
					$exportsettings['contact_comment'] = $result->contact_comment;
			
		
				}else{

					$resultss = array (
						"status" => "failure",
						"result" => 'Δεν υπαρχει εγγραφή για: '.$id_referrence,
					);
					echo json_encode ( $resultss );
					wp_die();

				}	
			

				$resultss = array (
					"status" => "success",
					"message" => '',
					"exportsettings" => $exportsettings,
					"result" => ''
				);
		
				echo json_encode ( $resultss );
					
					

		
			}
		
			// ajax handlers must die
			wp_die();
		}	


		public function iben_elta_compine_vouchers_ajax_handler(){
			
			if (isset($_POST['iben_data'])) {
				
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];			

				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$order = wc_get_order(  absint( $data['order_id'] ) );

				if ( !is_a( $order, 'WC_Order' ) ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Order does not exist', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();	
				}		      
				
				$order_id  = $order->get_id();

				
				$get_vouchers = iben_elta_get_Array_Vouchers_from_id_order($order_id);
				
				$errorpdf = '';

				if( empty($get_vouchers) || sizeof( $get_vouchers ) == 0 ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Vouchers doesnt exist on DB', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();

				}else{

					
					$count_vouchers = count($get_vouchers['elta_delivery_vouchers']) ;

					require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
					$pdf = new \Clegginabox\PDFMerger\PDFMerger;
					$output = '';

					foreach ($get_vouchers['elta_delivery_vouchers'] as $item) {
						
						if ($item != ''){
							$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($item), iben_elta_get_account_from_delivery_voucher($item) );

							if ( $pdf_created['state'] !== true){

								$errorpdf.= '('.$item.') <br/>'.esc_html__('This PDF didnt created. You have to Regenerate it!', 'iben-woo-elta-courier').' <span class="iben_elta_pdf_error">'.$pdf_created['result'].'</span>';
							}else{
								
								$pdf->addPDF($pdf_created['result'], 'all');

							}
						}
					}//foreach


					$output.= $this->iben_display_html_vouchers('time', json_encode($get_vouchers['elta_delivery_vouchers']), $order_id );
					
					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';	

					if ($count_vouchers > 1 && $errorpdf == '' ){
						
						$random_string = $this->iben_generateRandomString();
						$name_pdf = $order_id.$random_string;

						$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
						$pdf->merge('file', $merge, 'P');

						$output .= iben_elta_Compined_Voucher_Link($name_pdf);
			
						$wpdb->update( 
							$table_name, 
							array( 
								'elta_delivery_compined_vouchers' => $name_pdf
							), 
							array( 'id_order' => $order_id, 'id_elta_reason' => 1 )
						);
						
						$current_user = wp_get_current_user();
						$default2 = array(							
							'logged_user' => esc_html( $current_user->user_login),
							'id_user' => get_current_user_id(),
							'id_arrays' => json_encode($order_id),
							'voucher_arrays' => json_encode($get_vouchers['elta_delivery_vouchers']),
							'compined_pdf' => $name_pdf,
							'time' => current_time('timestamp')
						);
						$table_name2 = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
						$result = $wpdb->insert($table_name2, $default2);

					}


					if ($count_vouchers > 0){

						$wpdb->update( 
							$table_name, 
							array( 
								'pdf_deleted' => 0
							), 
							array( 'id_order' => $order_id, 'id_elta_reason' => 1 )
						);
					}
		
					$result = $output;

					if (isset($errorpdf)){
						$result = $output.'<br/>'.$errorpdf;
					}
		
					$message = 'Generated';

					$resultss2 = array (
						"status" => "success",
						"message" => $message,
						"result" => $result
					);
			
					echo json_encode ( $resultss2 );


				}//if $get_vouchers exists

			
			}

			wp_die();
		}

		
		public function iben_elta_compine_returned_vouchers_ajax_handler(){
			
			if (isset($_POST['iben_data'])) {
				
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];			

				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$order = wc_get_order(  absint( $data['order_id'] ) );

				if ( !is_a( $order, 'WC_Order' ) ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Order does not exist', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();	
				}
				
				$order_id  = $order->get_id();
				
				
				$get_vouchers = iben_elta_get_Array_Returned_Vouchers_from_id_order($order_id);
				

				if( empty($get_vouchers) || sizeof( $get_vouchers ) == 0 ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Vouchers doesnt exist on DB', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();					
				}else{

					$count_vouchers = count($get_vouchers['elta_returned_vouchers']) ;

					require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
					$pdf = new \Clegginabox\PDFMerger\PDFMerger;

					$output = '';
					$errorpdf = '';
					foreach ($get_vouchers['elta_returned_vouchers'] as $item) {

						if ($item != ''){


							$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($item), iben_elta_get_account_from_returned_voucher($item));

							if ( $pdf_created['state'] !== true){
								
								$errorpdf.= '('.$item.') <br/>'.esc_html__('This PDF didnt created. You have to Regenerate it!', 'iben-woo-elta-courier').' <span class="iben_elta_pdf_error">'.$pdf_created['result'].'</span>';
								
							}else{

								$pdf->addPDF($pdf_created['result'], 'all');

							}
						}
					}//foreach

					$output.= $this->iben_display_html_vouchers('time', json_encode($get_vouchers['elta_returned_vouchers']), $order_id );
					$result1 = $output;
					
					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';

					if ($count_vouchers > 1 && $errorpdf == '' ){

						$random_string = $this->iben_generateRandomString();
						$name_pdf = $order_id.$random_string;

						$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
						$pdf->merge('file', $merge, 'P');
						$output.= iben_elta_Compined_Voucher_Link($name_pdf);
			
						$wpdb->update( 
							$table_name, 
							array( 
								'elta_returned_compined_vouchers' => $name_pdf
							), 
							array( 'id_order' => $order_id, 'id_elta_reason' => 2 )
						);
						
						$current_user = wp_get_current_user();

						$default2 = array(							
							'logged_user' => esc_html( $current_user->user_login),
							'id_user' => get_current_user_id(),
							'id_arrays' => json_encode($order_id),
							'voucher_arrays' => json_encode($get_vouchers['elta_returned_vouchers']),
							'compined_pdf' => $name_pdf,
							'time' => current_time('timestamp')
						);
						$table_name2 = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
						$result = $wpdb->insert($table_name2, $default2);


					}

					if ($count_vouchers > 0){
						$wpdb->update( 
							$table_name, 
							array( 
								'pdf_deleted' => 0
							), 
							array( 'id_order' => $order_id, 'id_elta_reason' => 2 )
						);

					}
					
					if (isset($errorpdf)){
						$result1 = $output.'<br/>'.$errorpdf;
					}

					$message = 'Generated';
					$resultss = array (
						"status" => "success",
						"message" => $message,
						"result" => $result1
					);
			
					echo json_encode ( $resultss );


				}//if $get_vouchers exists

			
			}

			wp_die();
		}
		

		public function iben_elta_bulk_reprint_vouchers_ajax_handler(){
			
			if (isset($_POST['iben_data'])) {
				
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];			

				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$message = 'Generated';
				$output = '';

				// it secured by absint later
				$id_orders = $data['id_arrays'];


				if ( !isset($id_orders) || count($id_orders) == 0 && sizeof($id_orders) == 0 ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You have to check some orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				
				
				$compined_name = sanitize_text_field($data['compined_name']);
				
				
				// count of selected order ids
				$count_id_orders = count($id_orders) ;
				$count_loop = 0;

				$count_added = 0;

				require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
				$pdf = new \Clegginabox\PDFMerger\PDFMerger;
			
				$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$compined_name.'.pdf';

				$voucher_exist = false;

				$voucher_arrays = array();
				$errorpdf = '';
				
				foreach ($id_orders as $id_order) {
					
					$count_loop++;

					$counting = $count_loop.'/'.$count_id_orders;

					$get_vouchers = iben_elta_get_Array_Vouchers_from_id_order( $id_order );

					if( empty($get_vouchers) || sizeof( $get_vouchers ) == 0 || $get_vouchers == 0 ){
						

					}else{
		
						foreach ($get_vouchers['elta_delivery_vouchers'] as $voucher) {
		
							if ($voucher != ''){

								$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$voucher.".pdf";
								//if pdf doesnt exist, create it with WEBSERVICE
								if(!is_file($file)){

									$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF($voucher, iben_elta_get_account_from_delivery_voucher($voucher));

									if ( $pdf_created['state'] !== true){

										// DID NOT CREATED $pdf_created['result'];
										$errorpdf.= '('.$voucher.') '.esc_html__('This PDF didnt created. You have to Regenerate it!', 'iben-woo-elta-courier').' <span class="iben_elta_pdf_error">'.$pdf_created['result'].'</span>';
			
									}else{
										$count_added++;

										array_push($voucher_arrays, $pdf_created['voucher']);

										$pdf->addPDF($pdf_created['result'], 'all');

										$voucher_exist = true;
									}
								}else{
									$count_added++;

									array_push($voucher_arrays, $voucher);
									$pdf->addPDF($file, 'all');

									$voucher_exist = true;

								}

							}else{
								// Empty voucher here
							}
						}//foreach
						
					}

					if ($voucher_exist && $errorpdf == ''){
						$pdf->merge('file', $merge, 'P');
					}
					
				}//each id order

				$current_user = wp_get_current_user();

				if ($voucher_exist){

					$default = array(						
						'logged_user' => esc_html( $current_user->user_login),
						'id_user' => get_current_user_id(),
						'id_arrays' => json_encode($id_orders),
						'voucher_arrays' => json_encode($voucher_arrays),
						'compined_pdf' => $compined_name,
						'time' => current_time('timestamp')
					);

					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
					$result = $wpdb->insert($table_name, $default);

					// PDF compined

					$output1 = iben_elta_Compined_Voucher_Link($compined_name);	


					if (isset($errorpdf)){
						$output1 = $output1.'<br/>'.$errorpdf;
					}

				}else{

					// $output1[] = esc_html__('No vouchers generated!', 'iben-woo-elta-courier');
					$output1 = esc_html__('No vouchers generated!', 'iben-woo-elta-courier');

				}

				$resultss = array (
					"status" => "success",
					"message" => $message,
					"result" => $output1
				);
		
				echo json_encode ( $resultss );

			}

			wp_die();
		}

		public function iben_elta_tracing_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				
				$news = $this->ibenEltaWebServiceTracing( sanitize_text_field($data['voucher']), sanitize_text_field($data['ref_flag']) );

				
				if ($news['state'] === false){
					$message = esc_html__('Nothing to display!', 'iben-woo-elta-courier');
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $news['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}
				
				
				if ($news['state'] === true){
					
					$news = $news['result'];

			
					$delivered_name = $news->getPod_name();

					if ($delivered_name){
						
						$date = substr($news->getPod_date(), 6, 2).'-'.substr($news->getPod_date(), 4, 2).'-'.substr($news->getPod_date(), 0, 4);
						$time = substr($news->getPod_time(), 0, 2).':'.substr($news->getPod_time(), 2, 2);

						$result1 = '<div>
						<span class="title">'.esc_html__('Delivered!', 'iben-woo-elta-courier').'</span>
						<span class="name">'.$delivered_name.'</span>
						<span class="date">'.$date.'</span>
						<span class="time">'.$time.'</span></div>';

					}else{
						$route = '';
						$array = $news->getWeb_status();

						foreach ($array as $item){

							if( !$item->getWeb_status_title() ) {
								continue;
							}
				
							$route.= $item->getWeb_status_title().'<br/>';

							$route.= substr($item->getWeb_date(), 6, 2).'-'.substr($item->getWeb_date(), 4, 2).'-'.substr($item->getWeb_date(), 0, 4).' '.substr($item->getWeb_time(), 0, 2).':'.substr($item->getWeb_time(), 2, 2).'<br/>';
						}

						$result1 = $route;
					}


					
					$message = '';
		
					$resultss = array (
						"status" => "success",
						"message" => $message,
						"result" => $result1
					);
			
					echo json_encode ( $resultss );			

				} 


		
			}
		
			// ajax handlers must die
			wp_die();

		}		

		public function iben_elta_check_delivery_list_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$order_id = sanitize_text_field($data['id_order']);						
				$id_elta_reason = sanitize_text_field( $data['id_elta_reason'] );
				$id_elta_account = sanitize_text_field( $data['id_elta_account'] );
				
				$update_id = sanitize_text_field( $data['update_id'] );


				if (isset( $id_elta_reason ) &&  $id_elta_reason == '1'){

					$get_vouchers = iben_elta_get_Array_Vouchers_from_id( $update_id );

					$voucher_to_check = $get_vouchers['0'];

				}else if (isset( $id_elta_reason ) &&  $id_elta_reason == '2'){

					$get_vouchers = iben_elta_get_Array_Returned_Vouchers_from_id( $update_id );
					// 0 =return voucher, 1 = 1 st normal voucher
					$voucher_to_check = $get_vouchers['1'];

				}	

				if( empty($get_vouchers) || sizeof( $get_vouchers ) == 0 ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Vouchers doesnt exist on DB', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				$news = $this->ibenEltaWebServiceTracing( $voucher_to_check, '1', $id_elta_account );

				if ($news['state'] === false){
					$message = esc_html__('Nothing to display!', 'iben-woo-elta-courier');


					$alternative_canceled = false;
					if ( strpos($news['result'], 'Λάθος SYDETA') !== false ) {
						$alternative_canceled = true;
					}

					if ($alternative_canceled){
						global $wpdb;
						$table_name = $wpdb->prefix . 'iben_elta_vouchers';
						$updatedata = [ 'deleted' => '1', 'last_check_delivered_time' => current_time('timestamp')];					
						$where = [ 'id' => $update_id ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );		
					}
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $news['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				
				if ($news['state'] === true){


					$news = $news['result'];
					$alternative_delievered = false;
					$risk_returning = false;
					$returned = false;
					$delivered_name = $news->getPod_name();

					$getWeb_status = $news->getWeb_status();
					$getWeb_status = $getWeb_status['0'];
					$getWeb_status = $getWeb_status->getWeb_status_title();
		

					$array = array('ΑΠΟΣΤΟΛΗ ΠΑΡΑΔΟΘΗΚΕ-ΧΩΡΙΣ ΣΤΟΙΧΕΙΟ ΠΑΡΑΔΟΣΗΣ');

					if (in_array($getWeb_status, apply_filters('iben_elta_filter_delivery_array_strings', $array), true) === true){
						$alternative_delievered = true;
					}

					$array_returning = array('ΕΙΣΑΓΩΓΗ ΣΤΗ ΘΥΡΙΔΑ', 'ΕΙΣΑΓΩΓΗ ΘΥΡΙΔΑ', 'ΠΑΡΑΛΗΠΤΗΣ ΕΚΤΟΣ - ΕΤΑΙΡΙΑ ΚΛΕΙΣΤΗ',  'Ο ΠΑΡΑΛΗΠΤΗΣ ΔΕΝ ΕΝΤΟΠΙΖΕΤΑΙ', 'ΠΑΡΑΛΗΠΤΗΣ ΔΕΝ ΕΝΤΟΠΙΖΕΤΑΙ', 'ΕΙΔΟΠΟΙΗΘΗΚΕ - ΕΙΣΑΓΩΓΗ ΣΤΟΝ ΣΤΑΘΜΟ');
					if (in_array($getWeb_status, apply_filters('iben_elta_filter_returning_array_strings', $array_returning), true) === true){
						$risk_returning = true;
					}
					
					$array_returned = array('ΕΠΙΣΤΡΟΦΗ ΣΤΟΝ ΑΠΟΣΤΟΛΕΑ', 'ΑΚΥΡΩΣΗ ΑΠΟ ΔΙΑΝΟΜΕΑ');
					if (in_array($getWeb_status, apply_filters('iben_elta_filter_returned_array_strings', $array_returned), true) === true){
						$returned = true;
					}					
					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';

					if ($delivered_name || $alternative_delievered){
						
						$date = substr($news->getPod_date(), 6, 2).'-'.substr($news->getPod_date(), 4, 2).'-'.substr($news->getPod_date(), 0, 4);
						$time = substr($news->getPod_time(), 0, 2).':'.substr($news->getPod_time(), 2, 2);
						

						$updatedata = [ 'delivered_name' => $delivered_name, 'delivered_time' => $time, 'delivered_date' => $date, 'delivered' => '1']; 						 
						$where = [ 'id' => $update_id ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );
						
						do_action( 'iben_elta_check_delivery_hook_after_delivered', $order_id, $update_id);

						$result1 = '<div>
						<span class="title">'.esc_html__('Delivered!', 'iben-woo-elta-courier').'</span>
						<span class="name">'.esc_html__('Reciever:', 'iben-woo-elta-courier').' '.$delivered_name.'</span>
						<span class="date">'.$date.'</span>
						<span class="time">'.$time.'</span>
						</div>';
						
					}elseif ( $risk_returning){

						$updatedata = [ 'delivered' => '3',  'last_check_delivered_time' => current_time('timestamp')]; 						 
						$where = [ 'id' => $update_id ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );
	
						do_action( 'iben_elta_check_delivery_hook_after_riskreturning', $order_id, $update_id);
						$result1 = '<span class="iben_status_notdelievered">'.$getWeb_status.'</span>';
					}elseif ( $returned){

						$updatedata = [ 'delivered' => '2',  'last_check_delivered_time' => current_time('timestamp')]; 						 
						$where = [ 'id' => $update_id ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );
	
						do_action( 'iben_elta_check_delivery_hook_after_returned', $order_id, $update_id);
						$result1 = '<span class="iben_status_notdelievered">'.$getWeb_status.'</span>';
					}else{
						
						global $wpdb;
						$table_name = $wpdb->prefix . 'iben_elta_vouchers';

						$updatedata = [ 'delivered' => '-1', 'last_check_delivered_time' => current_time('timestamp')]; 					 
						$where = [ 'id' => $update_id ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );						
						
						$result1 = '<span class="iben_status_notdelievered">'. esc_html__( 'Not delivered yet', 'iben-woo-elta-courier' ).'</span>';
					}
					
					$message = '';
		
					$resultss = array (
						"status" => "success",
						"message" => $message,
						"result" => $result1
					);
			
					echo json_encode ( $resultss );

				}
				

		
			}
		
			// ajax handlers must die
			wp_die();

		}

		public function iben_elta_voucher_tracing_modal_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				$order_id = sanitize_text_field($data['id_order']);				
				$id_elta_reason = sanitize_text_field( $data['id_elta_reason'] );
				$id_elta_account = sanitize_text_field( $data['id_elta_account'] );
				


				if (isset( $id_elta_reason ) &&  $id_elta_reason == '1'){

					$get_vouchers = iben_elta_get_Array_Vouchers_from_id( absint($data['id_response']) );

					$voucher_to_check = $get_vouchers['0'];

				}else if (isset( $id_elta_reason ) &&  $id_elta_reason == '2'){

					$get_vouchers = iben_elta_get_Array_Returned_Vouchers_from_id( absint($data['id_response']) );
					// 0 =return voucher, 1 = 1 st normal voucher
					$voucher_to_check = $get_vouchers['1'];

				}	

				if( empty($get_vouchers) || sizeof( $get_vouchers ) == 0 ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Vouchers doesnt exist on DB', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				$news = $this->ibenEltaWebServiceTracing( $voucher_to_check, '1', $id_elta_account );

				if ($news['state'] === false){
					$message = esc_html__('Nothing to display!', 'iben-woo-elta-courier');
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $news['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				
				if ($news['state'] === true){
					
					$news = $news['result'];
					$alternative_delievered = false;
					$risk_returning = false;
					$returned = false;
					$route = '';
					
					$delivered_name = $news->getPod_name();

					$getWeb_status = $news->getWeb_status();
					$getWeb_status = $getWeb_status['0'];
					$getWeb_status = $getWeb_status->getWeb_status_title();					

					$array = array('ΑΠΟΣΤΟΛΗ ΠΑΡΑΔΟΘΗΚΕ-ΧΩΡΙΣ ΣΤΟΙΧΕΙΟ ΠΑΡΑΔΟΣΗΣ');

					if (in_array($getWeb_status, apply_filters('iben_elta_filter_delivery_array_strings', $array), true) === true){
						$alternative_delievered = true;
					}

					$array_returning = array('ΕΙΣΑΓΩΓΗ ΣΤΗ ΘΥΡΙΔΑ', 'ΕΙΣΑΓΩΓΗ ΘΥΡΙΔΑ', 'ΠΑΡΑΛΗΠΤΗΣ ΕΚΤΟΣ - ΕΤΑΙΡΙΑ ΚΛΕΙΣΤΗ',  'Ο ΠΑΡΑΛΗΠΤΗΣ ΔΕΝ ΕΝΤΟΠΙΖΕΤΑΙ', 'ΠΑΡΑΛΗΠΤΗΣ ΔΕΝ ΕΝΤΟΠΙΖΕΤΑΙ', 'ΕΙΔΟΠΟΙΗΘΗΚΕ - ΕΙΣΑΓΩΓΗ ΣΤΟΝ ΣΤΑΘΜΟ');
					if (in_array($getWeb_status, apply_filters('iben_elta_filter_returning_array_strings', $array_returning), true) === true){
						$risk_returning = true;
					}
					
					$array_returned = array('ΕΠΙΣΤΡΟΦΗ ΣΤΟΝ ΑΠΟΣΤΟΛΕΑ', 'ΑΚΥΡΩΣΗ ΑΠΟ ΔΙΑΝΟΜΕΑ');
					if (in_array($getWeb_status, apply_filters('iben_elta_filter_returned_array_strings', $array_returned), true) === true){
						$returned = true;
					}

					global $wpdb;
					$table_name = $wpdb->prefix . 'iben_elta_vouchers';

					if ($delivered_name || $alternative_delievered){
						
						$date = substr($news->getPod_date(), 6, 2).'-'.substr($news->getPod_date(), 4, 2).'-'.substr($news->getPod_date(), 0, 4);
						$time = substr($news->getPod_time(), 0, 2).':'.substr($news->getPod_time(), 2, 2);

						$route.= '<div>
						<span class="title">'.esc_html__('Delivered!', 'iben-woo-elta-courier').'</span>
						<span class="name">'.$delivered_name.'</span>
						<span class="date">'.$date.'</span>
						<span class="time">'.$time.'</span></div>';

						$updatedata = [ 'delivered_name' => $delivered_name, 'delivered_time' => $time, 'delivered_date' => $date, 'delivered' => '1']; 						 
						$where = [ 'id' => $data['id_response'] ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );						
						
						do_action( 'iben_elta_check_delivery_hook_after_delivered', $order_id, $data['id_response']);

					}elseif ( $risk_returning){

						$updatedata = [ 'delivered' => '3',  'last_check_delivered_time' => current_time('timestamp')]; 						 
						$where = [ 'id' => $data['id_response'] ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );
	
						do_action( 'iben_elta_check_delivery_hook_after_riskreturning', $order_id, $data['id_response']);
						
					}elseif ( $returned){

						$updatedata = [ 'delivered' => '2',  'last_check_delivered_time' => current_time('timestamp')]; 						 
						$where = [ 'id' => $data['id_response'] ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );
	
						do_action( 'iben_elta_check_delivery_hook_after_returned', $order_id, $data['id_response']);
						
					}else{
						// NOT DELIVERED				
						$updatedata = [ 'delivered' => '-1', 'last_check_delivered_time' => current_time('timestamp')]; 						 
						$where = [ 'id' => $data['id_response'] ];
						$db_update = $wpdb->update( $table_name, $updatedata, $where );
					}

						$array = $news->getWeb_status();

						foreach ($array as $item){

							if( !$item->getWeb_status_title() ) {
								continue;
							}
				
							$route.= $item->getWeb_status_title().'<br/>';
				
							$route.= substr($item->getWeb_date(), 6, 2).'-'.substr($item->getWeb_date(), 4, 2).'-'.substr($item->getWeb_date(), 0, 4).'-'.substr($item->getWeb_time(), 0, 2).':'.substr($item->getWeb_time(), 2, 2).'<br/>';
						}

						$result1 = $route;

					
					$message = '';
		
					$resultss = array (
						"status" => "success",
						"message" => $message,
						"result" => $result1
					);
			
					echo json_encode ( $resultss );			

				} 
				

		
			}
		
			// ajax handlers must die
			wp_die();

		}


		public function iben_elta_voucher_info_modal_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				
				$id = absint($data['id_response']);

	
				global $wpdb;
				$table_name = $wpdb->prefix . 'iben_elta_vouchers';
				
				$row = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $id));
				
				$iben_elta_meta_values = maybe_unserialize($row->elta_voucher_info);


				// if( empty($get_vouchers) || sizeof( $get_vouchers ) == 0 ){

				// 	$resultss = array (
				// 		"status" => "failure",
				// 		"result" => esc_html__( 'Vouchers doesnt exist on DB', 'iben-woo-elta-courier' )
				// 	);
				// 	echo json_encode ( $resultss );
				// 	wp_die();
				// }

				
				$result1 = '';
				if ( isset($iben_elta_meta_values) && $iben_elta_meta_values != '' ){
					$result1 = '<h3>'.esc_html__('Voucher info', 'iben-woo-elta-courier').'</h3>';

					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Full name', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['fullname'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Address', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['address'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Region', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['region'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Postal Code', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['postal_code'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Phone1', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['phone1'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Phone2', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['phone2'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Comments', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['comments'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Cash delivery', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['cash_delivery'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Weight', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['weight'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Number of packages', 'iben-woo-elta-courier').':</span> '.$iben_elta_meta_values['packages_number'].'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Service', 'iben-woo-elta-courier').':</span> '.( isset($iben_elta_meta_values['service']) && $iben_elta_meta_values['service']  == 1 ? '201' : '204' ).'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Special management', 'iben-woo-elta-courier').':</span> '.( isset($iben_elta_meta_values['sur_1']) && $iben_elta_meta_values['sur_1'] == 0 ? esc_html__('No', 'iben-woo-elta-courier') : esc_html__('Yes', 'iben-woo-elta-courier')).'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Delivery Time', 'iben-woo-elta-courier').':</span> '. ( isset($iben_elta_meta_values['sur_2']) && $iben_elta_meta_values['sur_2'] == 0 ? esc_html__('No', 'iben-woo-elta-courier') : esc_html__('Yes', 'iben-woo-elta-courier')).'</span>';
					$result1 .='<span class="iben_elta_voucher_modal_row" style="display:block;" ><span class="iben_elta_voucher_modal_title">'.esc_html__('Delivery Saturday', 'iben-woo-elta-courier').':</span> '.( isset($iben_elta_meta_values['sur_3']) && $iben_elta_meta_values['sur_3'] == 0 ? esc_html__('No', 'iben-woo-elta-courier') : esc_html__('Yes', 'iben-woo-elta-courier')).'</span>';

				}else{
					$result1 = '<h3>'.esc_html__('No available info', 'iben-woo-elta-courier').'</h3>';
				}

						
				$message = '';
					
				$resultss = array (
					"status" => "success",
					"message" => $message,
					"result" => $result1
				);
		
			echo json_encode ( $resultss );
		
			}
		
			// ajax handlers must die
			wp_die();

		}


		public function iben_elta_voucher_create_modal_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				
				$order = wc_get_order( absint( $data['id_order'] ) );


				if ( !is_a( $order, 'WC_Order' ) ){

					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'Order does not exist', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();	
				}
				
				$order_id  = $order->get_id();

				$result1 = '';
				$result1 .= $this->iben_elta_meta_box_html($order, true);
						
				$message = '';
					
				$resultss = array (
					"status" => "success",
					"message" => $message,
					"result" => $result1
				);
		
			echo json_encode ( $resultss );
		
			}
		
			// ajax handlers must die
			wp_die();

		}


		public function iben_elta_voucher_create_single_modal_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				
				$result1 = '';
				$result1 .= $this->iben_elta_meta_box_html_single();
						
				$message = '';
					
				$resultss = array (
					"status" => "success",
					"message" => $message,
					"result" => $result1
				);
		
			echo json_encode ( $resultss );
		
			}
		
			// ajax handlers must die
			wp_die();

		}		
		
		/**
		 * When an order created, add order ID on queue
		 * 
		 * @since     1.6.3
		 */	
		public function iben_elta_station_onload_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
				$message = '';
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				} 

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				
				$news = $this->ibenEltaWebServiceStation( sanitize_text_field($data['postal_code']) );

				if ($news['state'] === false){
					$message = esc_html__('Nothing to display!', 'iben-woo-elta-courier');
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $news['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				if ($news['state'] === true){
					
					$news = $news['result'];
			
						// $result1 = '';
			 			
						// $result1.= '<span class = "iben_station_item">'.( $news->getWeb_company() !== '' ? $news->getWeb_company() : '' ).'</span>';
						// $result1.= '<span class = "iben_station_item">'.( $news->getWeb_title() !== '' ? $news->getWeb_title() : '' ).'</span>';
						// $result1.= '<span class = "iben_station_item">'.( $news->getWeb_address() !== '' ? $news->getWeb_address() : '' ).'</span>';
						// $result1.= '<span class = "iben_station_item">'.( $news->getWeb_area() !== '' ? $news->getWeb_area() : '' ).'</span>';
						// $result1.= '<span class = "iben_station_item">'.( $news->getWeb_tel1() !== '' ? $news->getWeb_tel1() : '' ).'</span>';
							
						$result1 = '';
						$result1.= $news->getWeb_title().'<br/><span class="iben_elta_station_info">';
						$result1.= $news->getWeb_address().', ';
						$result1.= $news->getWeb_area().', ';
						$result1.= $news->getWeb_tel1().'</span>';						
				
						$resultss = array (
							"status" => "success",
							"message" => $message,
							"result" => $result1
						);
			
					echo json_encode ( $resultss );

				}
		
			}
		
			// ajax handlers must die
			wp_die();

		}


		public function iben_elta_station_ajax_handler(){

			if (isset($_POST['iben_data'])) {
		
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];
				$message = '';
		
				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				} 

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				
				$news = $this->ibenEltaWebServiceStation( sanitize_text_field($data['postal_code']) );

				if ($news['state'] === false){
					$message = esc_html__('Nothing to display!', 'iben-woo-elta-courier');
					
					$resultss = array (
						"status" => "failure",
						"message" => $message,
						"result" => $news['result'],
					);
					echo json_encode ( $resultss );
					wp_die();
				}


				if ($news['state'] === true){
					
					$news = $news['result'];
			
						$result1 = '';
			 			
						$result1.= '<span class = "iben_station_item">'.( $news->getWeb_company() !== '' ? $news->getWeb_company() : '' ).'</span>';
						$result1.= '<span class = "iben_station_item">'.( $news->getWeb_title() !== '' ? $news->getWeb_title() : '' ).'</span>';
						$result1.= '<span class = "iben_station_item">'.( $news->getWeb_address() !== '' ? $news->getWeb_address() : '' ).'</span>';
						$result1.= '<span class = "iben_station_item">'.( $news->getWeb_area() !== '' ? $news->getWeb_area() : '' ).'</span>';
						$result1.= '<span class = "iben_station_item">'.( $news->getWeb_tel1() !== '' ? $news->getWeb_tel1() : '' ).'</span>';
								
				
						$resultss = array (
							"status" => "success",
							"message" => $message,
							"result" => $result1
						);
			
					echo json_encode ( $resultss );

				}
		
			}
		
			// ajax handlers must die
			wp_die();

		}		

		public function iben_display_html_vouchers($time, $vouchers, $order_id, $hide_compine = false, $hide_time = false){
			
			$return_string = '';
			if (!$hide_time){
				$return_string.= '<span class="iben_elta_order_voucher_time normal">'.date_i18n(get_option('date_format'), $time) .' | '. date_i18n(get_option('time_format'), $time).'</span>';
			}
			$vouchers = json_decode($vouchers);
			
			if( !empty($vouchers) && sizeof( $vouchers ) != 0 ) {
				$count = 0;

				foreach ($vouchers as $voucher) {

					if ($count == 0){
						$tracking_code = $voucher;
					}
					if ($voucher != ''){
						

						$file = IBEN_WOO_ELTA_UPLOAD_FOLDER.$voucher.".pdf";
						
						if(is_file($file)){
							$return_string.= iben_elta_Voucher_Link($voucher);
							
						}else{
							$return_string.= '<span class="iben_elta_order_voucher_item normal" style="display: block;">'.$voucher.'</span>';
						}
						
						$count++;
					}
				}

			}
			
			return $return_string;
		}


		public function iben_display_html_returned_vouchers($time, $vouchers, $order_id, $hide_compine = false){
			$return_string = '';
			$return_string.= '<span class="iben_elta_order_voucher_time returned">'.date_i18n(get_option('date_format'), $time) .' | '. date_i18n(get_option('time_format'), $time).'</span>';
			
			$vouchers = json_decode($vouchers);
			
			if( !empty($vouchers) && sizeof( $vouchers ) != 0 ) {
				$count = 0;

				foreach ($vouchers as $voucher) {
					if ($voucher != ''){
						$return_string.= '<span class="iben_elta_order_voucher_item returned" style="display: block;">'.$voucher.'</span>';
						$count++;
					}
				}
			}

			return $return_string;
		}

			
		/**
		 *
		 * @param string[] $columns
		 * @return string[] $new_columns
		 */
		function iben_elta_add_order_vouchers_column_header( $columns ) {


			$new_columns = array();

			foreach ( $columns as $column_name => $column_info ) {

				$new_columns[ $column_name ] = $column_info;

				if ( 'order_status' === $column_name ) {
					$new_columns['iben_elta_order_voucher'] = esc_html__( 'ELTA Voucher', 'iben-woo-elta-courier' );
				}
			}

			return $new_columns;
		}


		function iben_elta_add_order_vouchers_column_content( $column, $order_id ) {

			if ('iben_elta_order_voucher' !== $column) return;

			$order = wc_get_order( $order_id );

			if ( !is_a( $order, 'WC_Order' )  ){
				return;
			}


			$billing_country = $order->get_billing_country();
			$shipping_country = $order->get_shipping_country();			

			if ( $billing_country != $shipping_country && $shipping_country != '' ){
				$country = $shipping_country;
			}else{
				$country = $billing_country;
			}

			$order_status = $order->get_status();


			$list_countries = array_keys($this->countries);
			$list_countries = apply_filters('iben_elta_courier_order_metabox_display_countries', $list_countries);

			$except_list_order_states =  array("pending", "cancelled", "failed" );
			$except_list_order_states = apply_filters('iben_elta_courier_order_metabox_not_display_order_status', $except_list_order_states);			

			if ( ($country == '' || in_array($country, $list_countries) ) && !in_array($order_status, $except_list_order_states) ) {

				add_thickbox(); 
		
				$vouchers = '';

				$get_vouchers = iben_elta_get_Vouchers_from_id_order( $order->get_id() );

				foreach ($get_vouchers as $voucher){
					echo '<span class = "iben_elta_order_list_vouchers_block">';

						echo $this->iben_display_html_vouchers($voucher->time, $voucher->elta_delivery_vouchers, $order->get_id());
		
					echo '</span>';
		
					echo '<span class = "iben_elta_voucherslist_status_response_block">';
		
					if ($voucher->delivered == 1){
						echo '<b>'.esc_html__('Delivered', 'iben-woo-elta-courier').'</b>';
						echo '<br/>';
						echo esc_html($voucher->delivered_date.' '.$voucher->delivered_time);
						echo '<br/>';
					}else{
			
						?>
						<span class="iben_elta_voucherslist_status_response_c" id="iben_elta_voucherslist_status_response<?php echo absint( $voucher->id); ?>">
							<?php
							if ($voucher->delivered == -1){
								echo '<p>'.esc_html__('Hasn\'t delivered yet', 'iben-woo-elta-courier').'</p>';
							}
							?>
							<a href='#' title="<?php esc_html_e('Check delivery', 'iben-woo-elta-courier') ?>" alt="<?php esc_html_e('Check delivery', 'iben-woo-elta-courier') ?>" class="iben_elta_vouchers_status_list" data-id_elta_account = "<?php echo absint( $voucher->id_elta_account); ?>" data-id_elta_reason = "<?php echo absint( $voucher->id_elta_reason); ?>" data-id_order="<?php echo absint( $voucher->id_order); ?>" data-updateid="<?php echo absint( $voucher->id); ?>"><?php  esc_html_e('Check status', 'iben-woo-elta-courier'); ?></a>
						</span>
				
		
					<?php
					}
					?>
						<a href="#TB_inline&width=400&height=400&inlineId=iben_elta_order_voucher_modal-<?php echo absint( $voucher->id); ?>" data-id_elta_reason = "<?php echo absint( $voucher->id_elta_reason); ?>"  data-id_order = "<?php echo absint( $voucher->id_order); ?>"  data-id_response = "<?php echo absint( $voucher->id); ?>" class="thickbox iben_elta_voucher_info_modal_trigger" title = "<?php esc_html_e('Voucher info', 'iben-woo-elta-courier'); ?>"><?php  esc_html_e('Voucher info', 'iben-woo-elta-courier'); ?></a>

						<div class="iben_elta_order_voucher_modal" id="iben_elta_order_voucher_modal-<?php echo absint( $voucher->id); ?>" style="display:none;">
							<div class="iben_elta_voucher_info_modal_response" id ="iben_elta_voucher_info_modal_response<?php echo absint( $voucher->id); ?>"></div>
						</div>					

				<?php
				echo '</span>';
				}
				
				
				
				if (!$get_vouchers && apply_filters('iben_elta_filter_display_create_voucher', true, $order) ){
				?>

					<a href="#TB_inline=true&width=900&height=700&inlineId=iben_elta_order_create_voucher_modal-<?php echo absint( $order->get_id() ); ?>"  data-id_order = "<?php echo absint( $order->get_id() ); ?>" class="thickbox iben_elta_voucher_create_modal_trigger" title = "<?php  esc_html_e('Create voucher', 'iben-woo-elta-courier'); ?>"><?php esc_html_e('Create voucher', 'iben-woo-elta-courier'); ?></a>

					<div class= "iben_elta_order_create_voucher_modal" id="iben_elta_order_create_voucher_modal-<?php echo absint( $order->get_id() ); ?>" style="display:none;">
						<div class= "iben_elta_voucher_create_modal_response" id ="iben_elta_voucher_create_modal_response<?php echo absint( $order->get_id() ); ?>"></div>
					</div>	


				<?php
				}

				do_action( 'woo_column_after_elta', $order->get_id() );


			}
			

		}

		function iben_elta_voucher_columns_style() {
		
			$css = '.widefat .column-iben_elta_order_voucher { width: 12ch!important; }';
			wp_add_inline_style( 'woocommerce_admin_styles', $css );
		}	

		

		/**
		 * When an order created, add order ID on queue
		 * 
		 * @since     1.6.0
		 */			
		function iben_elta_add_order_queue( $order_id ){

			$order = wc_get_order( $order_id );


			$pass_orders = array('processing');

			$pass_orders = apply_filters('iben_elta_filter_add_order_status', $pass_orders);

			if ( !is_a( $order, 'WC_Order' ) ||  !in_array($order->get_status(), $pass_orders) ){
				return;
			}

			$parent_id = $order->get_parent_id();
		
			if ( $parent_id ) {
				$parent_id = $parent_id;
			}else{
				$parent_id = 0;
			}

			$default = array(				
				'id_order' => $order_id,
				'id_order_parent' => $parent_id,
				'status' => $order->get_status(),
				'time' => current_time('timestamp'),
			);

			global $wpdb;
			$table_name = $wpdb->prefix . 'iben_elta_voucher_schedule';	
			$result = $wpdb->insert($table_name, $default);


		}

		/**
		 * Display subcode field on WP user profile
		 * Run If Dokan compatibility exist or filter 'iben_elta_filter_display_subcode_user_profile' is true
		 * 
		 * @since     1.6.0
		 */

		function iben_elta_display_subcode_wp_user( $user ) {

			if ( ! current_user_can( 'manage_woocommerce' ) ) {
				return;
			}

			$user_subcode = get_user_meta($user->ID, 'iben_elta_user_subcode_meta', true);
			
			?>

			<h3><?php esc_html_e( 'ELTA Options for dokan', 'iben-woo-elta-courier' ); ?></h3>
	
			<table class="form-table">
				
				<tr>
					<th><?php esc_html_e( 'Vendor Sub code', 'iben-woo-elta-courier' ); ?></th>
					<td>
						<input type="text" name="iben_elta_user_subcode_meta" class="regular-text" value="<?php echo esc_attr( $user_subcode ); ?>">
					</td>
				</tr>
				<?php
					wp_nonce_field( 'iben_elta_subcode_update_user_profile_info', 'iben_elta_subcode_update_user_profile_info_nonce' );
				?>
				</table>
		
			<?php
		}

		
		/**
		 * Save subcode field on WP user profile
		 * Run If Dokan compatibility exist or filter 'iben_elta_filter_display_subcode_user_profile' is true
		 * 
		 * @since     1.6.0
		 */
		function iben_elta_save_subcode_wp_user( $user_id ) {

			$post_data = wp_unslash( $_POST );

			if ( isset( $post_data['iben_elta_subcode_update_user_profile_info_nonce'] ) && ! wp_verify_nonce( $post_data['iben_elta_subcode_update_user_profile_info_nonce'], 'iben_elta_subcode_update_user_profile_info' ) ) {
				return;
			}

			if ( !current_user_can( 'edit_user', $user_id ) ){
				return;
			}	

			update_user_meta( $user_id, 'iben_elta_user_subcode_meta', !empty( $post_data['iben_elta_user_subcode_meta'] ) ? $post_data['iben_elta_user_subcode_meta'] : '' );

		}


		/**
		 * Attach tracking info on admin email when hooked
		 *
		 * @since     1.6.0
		 */
		public function iben_elta_email_send_admin_tracking_info_handler( $order, $sent_to_admin, $plain_text, $email ) {

			if ( $email->id != 'new_order' || $sent_to_admin != 1 || !is_object($order)) {
				return;
			}

			$order_id  = $order->get_id();
			$parent_id = $order->get_parent_id();

			if ( $parent_id ) {
				$parent_id = $parent_id;
			}else{
				$parent_id = $order_id;
			}

			$vouchers = '';

			$get_vouchers = iben_elta_get_Array_Vouchers_from_id_order($parent_id);
			
			if( !empty($get_vouchers) || sizeof( $get_vouchers ) != 0 ){

				foreach ($get_vouchers['elta_delivery_vouchers'] as $item) {

					$file2 = IBEN_WOO_ELTA_UPLOAD_FOLDER.$item.".pdf";                            
					
					if ($item != '' && is_file($file2) ){	
						
						$vouchers.= ' <a href="'.IBEN_WOO_ELTA_UPLOAD_VOUCHER_URL.$item.'.pdf" target="_blank">'.$item.'</a> ';
					}else {

						$vouchers.= $item.' ';
					}
				}//foreach

				
			}//if $get_vouchers exists

			if ($vouchers){

				$vouchers = '<span>'.$vouchers.'</span>';
			}

			$message = '';

			$options = get_option( 'iben_elta_main_settings_fields' );

			if ( isset($options['iben_elta_main_settings_auto_createv_admin_body']) && $options['iben_elta_main_settings_auto_createv_admin_body'] != '' ){
				$message = $options['iben_elta_main_settings_auto_createv_admin_body'];
			}

			if ($vouchers){
			
				$search = array("[iben_elta_tracking_pdf]");
				$replacewith = array( $vouchers);

				$replaced_message = apply_filters( 'iben_elta_send_admin_tracking_message_before_filter', $message, $order );

				$message = str_replace($search, $replacewith, $replaced_message);

				$allowed_html = array(
					'a' => array(
						'href' => array(),
						'title' => array(),
						'alt' => array(),
					),
					'br' => array(),
					'em' => array(),
					'strong' => array(),
					'ul' => array(),
					'li' => array(),
					'ol' => array(),
					'p' => array(),
					'h1' => array(),
					'h2' => array(),
					'h3' => array(),
					'h4' => array(),
					'h5' => array(),
					'h6' => array(),
					'i' => array(),
				);

				$allowed_html = apply_filters( 'iben_email_admin_tracking_metabox_email_allowed_html_filter', $allowed_html, $order );

				$message = wp_kses($message, $allowed_html);

				echo $message;
			}

	
		}

		/**
		 * DOKAN Comaptibility 
		 * Send Vendor New Order - Attach tracking info on admin email when hooked
		 *
		 * @since     1.6.0
		 */		
		public function iben_elta_email_send_vendor_admin_dokan_tracking_info_handler( $order, $sent_to_admin, $plain_text, $email ) {

			if ( $email->id != 'dokan_vendor_new_order' || !is_object($order)) {
				return;
			}

			$order_id  = $order->get_id();

			$vouchers = '';
			
			$get_vouchers = iben_elta_get_Array_Vouchers_from_id_order($order_id);

			if( !empty($get_vouchers) || sizeof( $get_vouchers ) != 0 ){

				foreach ($get_vouchers['elta_delivery_vouchers'] as $item) {

					$file2 = IBEN_WOO_ELTA_UPLOAD_FOLDER.$item.".pdf";                            
					
					if ($item != '' && is_file($file2) ){		
						$vouchers.= '<a href="'.IBEN_WOO_ELTA_UPLOAD_VOUCHER_URL.$item.'.pdf" target="_blank">'.$item.' </a>';
					}else {

						$vouchers.= $item;
					}
				}//foreach

				
			}//if $get_vouchers exists

			if ($vouchers){
				$vouchers = '<span>'.$vouchers.'</span>';
			}

			$message = '';

			$options = get_option( 'iben_elta_main_settings_fields' );

			if ( isset($options['iben_elta_main_settings_auto_createv_admin_body']) && $options['iben_elta_main_settings_auto_createv_admin_body'] != '' ){
				$message = $options['iben_elta_main_settings_auto_createv_admin_body'];
			}

			if ($vouchers){
			
				$search = array("[iben_elta_tracking_pdf]");
				$replacewith = array( $vouchers);

				$replaced_message = apply_filters( 'iben_elta_send_dokan_tracking_message_before_filter', $message, $order );

				$message = str_replace($search, $replacewith, $replaced_message);

				$allowed_html = array(
					'a' => array(
						'href' => array(),
						'title' => array(),
						'alt' => array(),
					),
					'br' => array(),
					'em' => array(),
					'strong' => array(),
					'ul' => array(),
					'li' => array(),
					'ol' => array(),
					'p' => array(),
					'h1' => array(),
					'h2' => array(),
					'h3' => array(),
					'h4' => array(),
					'h5' => array(),
					'h6' => array(),
					'i' => array(),
				);

				$allowed_html = apply_filters( 'iben_email_dokan_tracking_metabox_email_allowed_html_filter', $allowed_html, $order );
				

				$message = wp_kses($message, $allowed_html);


				echo $message;
			}

	
		}	


		/**
		 * 
		 *
		 * @since     1.6.0
		 */

		function iben_elta_cron_create_auto_vouchers_hook(){

			global $wpdb;
			$table_name = $wpdb->prefix . 'iben_elta_voucher_schedule';

			$limit = 3;			
			$limit = apply_filters('iben_elta_filter_auto_create_vouchers_cron_limit', $limit);

			$queries = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name LIMIT %d", $limit));

			if ($queries){

				$main_settings = $this->main_settings;	

				foreach ($queries as $record){

					if ( $record->id_order ) {
						$order = wc_get_order( $record->id_order );
					}
					
					if ( !is_a( $order, 'WC_Order' )  ){
						continue;
					}

					//SKIP IF IS NOT processing STATUS
					if (!$order->has_status('processing')){

						$order->add_order_note( esc_html__( 'AUTO ELTA. This order skipped from creating because wasnt in processing status', 'iben-woo-elta-courier' )  );

						$table_name = $wpdb->prefix . 'iben_elta_voucher_schedule';
						$wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE id = %d", $record->id));
						return;
					}

					$order_id = $order->get_id();

					$table_name = $wpdb->prefix . 'iben_elta_voucher_schedule';
					$wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE id = %d", $record->id));
					
					// DOKAN compatibility here
					$sub_order_ids   = [];
					$parent_order_ids   = [];
					
					$has_suborder = $order->get_meta( 'has_sub_order' );

					// DOKAN compatibility here
					if ( $has_suborder ) {
						
						$parent_order_ids[] = $order_id;

						$sub_order_ids = get_children(
							[
								'post_parent' => $order->get_id(),
								'post_type' => 'shop_order',
								'fields' => 'ids',
							]
						);

					} else {
						$sub_order_ids[] = $order_id;
					}


					foreach($sub_order_ids as $order_id ) {

						$order = wc_get_order( $order_id );
								
						$user_subcode = '';

						//DOKAN compatibility here
						if (function_exists('dokan_get_seller_id_by_order')) {
							$seller_id = dokan_get_seller_id_by_order( $order_id );	
							
							$user_subcode = get_user_meta($seller_id, 'iben_elta_user_subcode_meta', true);
							
							if ($user_subcode != '' ){
								$user_subcode = $user_subcode;
							}
							
						}
						

						$parent_id = $order->get_parent_id();
			
						if ( $parent_id ) {
							$parent_id = $parent_id;
						}else{
							$parent_id = 0;
						}
			
						if (1==1){
							$first_name = '';
							if( $order->get_billing_first_name() != $order->get_shipping_first_name() && !empty($order->get_shipping_first_name()) ) {
								$first_name = $order->get_shipping_first_name();
							} else {
								$first_name = $order->get_billing_first_name();
							}
							
							$last_name = '';
							if( $order->get_billing_last_name() != $order->get_shipping_last_name() && !empty($order->get_shipping_last_name()) ) {
								$last_name = $order->get_shipping_last_name();
							} else {
								$last_name = $order->get_billing_last_name();
							}
							
							$company = '';
			
							if( $order->get_billing_company() != $order->get_shipping_company() && !empty($order->get_shipping_company()) ) {
								$company = $order->get_shipping_company();
							} else {
								$company = $order->get_billing_company();
							}
							$company = ( $company == '' )? '':' ('.$company.')';
							
			
							$fullname = $last_name.' '.$first_name.$company; 
							$fullname = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($fullname, 'UTF-8') : $fullname;
							$fullname = apply_filters( 'iben_elta_filter_automatic_fullname_metafield_order', $fullname, $order_id );
			
							$address_1 = '';
							$address_2 = '';
							if( $order->get_billing_address_1() != $order->get_shipping_address_1() && !empty($order->get_shipping_address_1()) ) {
								$address_1 = $order->get_shipping_address_1();
							} else {
								$address_1 = $order->get_billing_address_1();
							}
			
			
							if( $order->get_billing_address_2() != $order->get_shipping_address_2() && !empty($order->get_shipping_address_2()) ) {
								$address_2 = $order->get_shipping_address_2();
							} else {
								$address_2 = $order->get_billing_address_2();
							}
			
						
							$display_address1 = apply_filters( 'iben_elta_filter_automatic_display_address1', true);
							$display_address2 = apply_filters( 'iben_elta_filter_automatic_display_address2', true);
							
			
							$fulladdress = '';
			
							if ($display_address1){
								$fulladdress.= $address_1;
							}
			
							if ( isset($address_2)  && $address_2 != '' ){
			
								if (!$display_address1 && $display_address2){
									$fulladdress.= $address_2;
								}else{
									$fulladdress.= ', '.$address_2;
								}			
							}
							$fulladdress = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($fulladdress, 'UTF-8') : $fulladdress;
							$fulladdress = apply_filters( 'iben_elta_filter_automatic_fulladdress_metafield_order', $fulladdress, $order_id );
			
							$billing_country = $order->get_billing_country();
							$shipping_country = $order->get_shipping_country();
			
			
							if ( $billing_country != $shipping_country && $shipping_country != '' ){
								$country = $shipping_country;
							}else{
								$country = $billing_country;
							}
			
			
							$billing_state = $order->get_billing_state();
							$shipping_state = $order->get_shipping_state();
			
							if ( $billing_state != $shipping_state && $shipping_state != '' ){
								$state = $shipping_state;
							}else{
								$state = $billing_state;
							}
			
							if ($country){
								if ($state){
									$state =  WC()->countries->get_states( $country )[$state];
								}else{
									$state = '';
								}
							}else{
			
								if ($state){
									$state =  WC()->countries->get_states( 'GR' )[$state];
								}else{
									$state = '';
								}
			
							}
							if ($order->get_billing_city() != $order->get_shipping_city()){
								$city = $order->get_shipping_city();
							}else{
								$city = $order->get_billing_city();
							}
			
							
							$display_city = apply_filters( 'iben_elta_filter_automatic_display_city', true);
							$display_state = apply_filters( 'iben_elta_filter_automatic_display_state', true);
			
							$region = '';
			
							if ($display_city){
								$region.= $city;
							}
			
							if (!$display_city && $display_state){
								$region.= $state;
							}else{
								$region.= ', '.$state;
							}
							$region = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($region, 'UTF-8') : $region;
							$region = apply_filters( 'iben_elta_filter_automatic_region_metafield_order', $region, $order_id );
			
							$postcode = '';
							if( $order->get_billing_postcode() != $order->get_shipping_postcode() && !empty($order->get_shipping_postcode()) ) {
								$postcode = $order->get_shipping_postcode();
							} else {
								$postcode = $order->get_billing_postcode();
							}
							$postcode = str_replace(array(' ', '-'), '', $postcode ?? '');
							$postcode = apply_filters( 'iben_elta_filter_automatic_postcode_metafield_order', $postcode, $order_id );
			
							$phone1 = '';
			
							$billing_phone1 = apply_filters('iben_elta_phone_format', $order->get_billing_phone(), $order_id);
							$billing_phone1 = str_replace(array(' ', '-', '.', '+'), '', $billing_phone1 ?? '');

							$shipping_phone1 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_automatic_phone1_metafield_order', 'shipping_phone' ), true));
							$shipping_phone1 = apply_filters('iben_elta_phone_format', $shipping_phone1, $order_id);
							$shipping_phone1 = str_replace(array(' ', '-', '.', '+'), '', $shipping_phone1 ?? '');

							if ( $billing_phone1 != $shipping_phone1 && $shipping_phone1 != ''){
								$phone1 = $shipping_phone1;
							}else{
								$phone1 = $billing_phone1;
							}
							$phone1 = apply_filters( 'iben_elta_filter_automatic_phone1', $phone1, $order_id);
			

							$phone2 = '';
							$billing_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_automatic_phone2_metafield_order', 'billing_mobile' ), true));
							$billing_phone2 = apply_filters('iben_elta_phone_format', $billing_phone2, $order_id);

							$shipping_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_automatic_shipping_phone2_metafield_order', 'shipping_mobile' ), true));
							$shipping_phone2 = apply_filters('iben_elta_phone_format', $shipping_phone2, $order_id);
			
							if ( $billing_phone2 != $shipping_phone2 && $shipping_phone2 != '' ){
								$phone2 = $shipping_phone2;
							}else{
								$phone2 = $billing_phone2;
							}
			

							$phone2 = apply_filters( 'iben_elta_filter_automatic_phone2', $phone2, $order_id);
			
							$comments = $order->get_customer_note();
							$comments = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($comments, 'UTF-8') : $comments;
							$comments = apply_filters( 'iben_elta_filter_automatic_comments_metafield_order', $comments, $order_id );
							
			
							$payment = $order->get_payment_method();
							
							//compatible with WC cod and 3rd party COD with name as id_codfee
							$payments = array('cod', 'id_codfee');
							
							$payments_filter = array();
							$payments_filter = apply_filters( 'iben_elta_filter_cashdelivery_payments_order_metabox', $payments_filter );
							
							if(is_array($payments_filter) && !empty($payments_filter)){
								foreach ($payments_filter as $payment_filter){
									$payments[] = $payment_filter;
								}
							}



							if (in_array($payment, $payments, true)) {
								$cashdelivery = $order->get_total();	
							}else{
								$cashdelivery = '0'; 
							}			
							$cashdelivery = apply_filters( 'iben_elta_filter_automatic_cash_metafield_order', $cashdelivery, $order_id );
							
							
							$weight = 0;
							// check if default weight has been set and use this
							$default_weight = sanitize_text_field($main_settings['iben_elta_main_settings_default_weight']);
			
							if( isset( $default_weight ) && $default_weight > 0 ) {
								
								$weight = $default_weight;
			
							}else{
						
								$total_weight = 0;
								foreach( $order->get_items() as $item_id => $product_item ){
									$quantity = $product_item->get_quantity(); // get quantity
									$product = $product_item->get_product(); // get the WC_Product object
									if( $product ) {
										$product_weight = $product->get_weight(); // get the product weight
										$total_weight += floatval( ( $product_weight ? $product_weight : 0 ) * $quantity );
										}
									
								}
								$weight = round($total_weight, 3);

								$unit = get_option('woocommerce_weight_unit');

								if ($unit == 'g'){
									$weight = $weight/1000;
								}
							}
							
							$weight = apply_filters( 'iben_elta_filter_automatic_weight_metafield_order', $weight, $order_id );
			
			
							$parcels = '1'; 
			
							$ref = '';
							if (isset( $main_settings['iben_elta_main_settings_enable_ref_order'] ) ){
								$ref = $order->get_order_number();
							}			
							$ref = apply_filters( 'iben_elta_filter_automatic_ref_metafield_order', $ref, $order_id );
						}
			
						
						$types['fullname'] = sanitize_text_field( $fullname ); 
						$types['address'] = sanitize_text_field( $fulladdress );
						$types['region'] = sanitize_text_field( $region );						
						$types['postal_code'] = sanitize_text_field( $postcode );
						$types['phone1'] = sanitize_text_field( $phone1 );
						$types['phone2'] = sanitize_text_field( $phone2 );
						$types['service'] = sanitize_text_field(1);
						$types['weight'] = sanitize_text_field( $weight );
						$types['packages_number'] = sanitize_text_field( $parcels );
						$types['subcode'] = sanitize_text_field( $user_subcode ); 

						$types['comments'] = sanitize_text_field( $comments );
						$types['cash_delivery'] = sanitize_text_field( $cashdelivery );
						$types['sur_1'] = sanitize_text_field(0);
						$types['sur_2'] = sanitize_text_field(0);
						$types['sur_3'] = sanitize_text_field(0);
						$types['ref'] = sanitize_text_field($ref);
						$types['second_account'] = sanitize_text_field( 1 );
						
						$types = apply_filters( 'iben_elta_filter_auto_create_types_order_metabox', $types, $order_id );
						$webservice_run = $this->ibenEltaWeServiceCreateVoucher($types);
												
			
						// success! Voucher has been created here
						if ($webservice_run['state'] === true){
							
							$webservice_run = $webservice_run['result'];
						
							$getVg_child = array();
							array_push($getVg_child, $webservice_run->getVg_code());
							$main_voucher = $webservice_run->getVg_code();
			
							if ( !empty($webservice_run->getVg_child()) ){
								$child_vouchers = array_filter($webservice_run->getVg_child());
			
								if( !empty($child_vouchers) && sizeof($child_vouchers) != 0) {
			
									foreach ($child_vouchers as $voucher){
										array_push($getVg_child, $voucher);
									}
								
								}
							}
			
							$default = array(							
								'logged_user' => 'Automatic',
								'id_user' => '-1',
								'id_order' => $order_id,
								'id_order_parent' => $parent_id,
								'id_elta_reason' => 1,
								'elta_returned_vouchers' => '',
								'elta_delivery_vouchers' => json_encode($getVg_child),
								'elta_pdf_type' => $this->iben_elta_main_settings_papersize,
								'elta_voucher_info' => maybe_serialize($types),
								'id_elta_account' => $types['second_account'],
								'duplicated_record' => 0,
								'time' => current_time('timestamp'),
								'plugin_version' => $this->version,
							);
			
			
							global $wpdb;
							$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
							$result = $wpdb->insert($table_name, $default);
					
			
							if ($parent_id){
			
								$default2 = array(									
									'logged_user' => 'Automatic',
									'id_user' => '-1',
									'id_order' => $parent_id,
									'id_order_parent' => -1,
									'id_elta_reason' => 1,
									'elta_returned_vouchers' => '',
									'elta_delivery_vouchers' => json_encode($getVg_child),
									'elta_pdf_type' => $this->iben_elta_main_settings_papersize,
									'elta_voucher_info' => maybe_serialize($types),
									'id_elta_account' => $types['second_account'],
									'duplicated_record' => 0,
									'time' => current_time('timestamp'),
									'plugin_version' => $this->version
								);
				
								global $wpdb;
								$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
								$result = $wpdb->insert($table_name, $default2);
			
							}

							do_action( 'iben_elta_hook_voucher_created', $order_id, $main_voucher );
			
							$order->update_meta_data( '_iben_elta_voucher_data', $types );
							$order->update_meta_data( '_iben_elta_voucher_ship_number', $main_voucher );

							$note = '';
							$get_vouchers = iben_elta_get_Vouchers_from_id_order($order_id);
			
							foreach ($get_vouchers as $voucher){
								$note .= $this->iben_display_html_vouchers($voucher->time, $voucher->elta_delivery_vouchers, $order_id, true);
							}
							if ( !empty($getVg_child) && $getVg_child != ''){

								$order->update_meta_data( '_iben_elta_voucher_generated_vouchers', json_encode($getVg_child) );
							}
		
			
							
							// always before SEND the email. 
							// if setting. Attach tracking info is enabled
							if ( !$order->meta_exists('iben_elta_tracking_order_notification') ){
							
								$is_enabled = isset( $main_settings['iben_elta_main_settings_email_completed'] ) ? (int)$main_settings['iben_elta_main_settings_email_completed'] : 0;	
								$is_default = isset( $main_settings['iben_elta_main_settings_email_checked_default_completed'] ) ? (int)$main_settings['iben_elta_main_settings_email_checked_default_completed'] : 0;	
	
								if($is_enabled && $is_default){
									$order->update_meta_data( 'iben_elta_tracking_order_notification', 1 );
								}
							}


							
			
							$errorpdf = '';
							//if setting. Create PDF is enabled
							if( isset( $main_settings['iben_elta_main_settings_auto_createv_create_pdf'] ) ) {

								$get_vouchers = iben_elta_get_Array_Vouchers_from_id_order($order_id);
			
								if( !empty($get_vouchers) || sizeof( $get_vouchers ) != 0 ){
					
									$count_vouchers = count($get_vouchers['elta_delivery_vouchers']) ;
				
									require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
									$pdf = new \Clegginabox\PDFMerger\PDFMerger;
									$output = '';
				
									foreach ($get_vouchers['elta_delivery_vouchers'] as $item) {
			
										$file2 = IBEN_WOO_ELTA_UPLOAD_FOLDER.$item.".pdf";                            
										
										if ($item != ''){
			
											$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($item), $types['second_account']);
				
											if ( $pdf_created['state'] === true){
				
												$pdf->addPDF($pdf_created['result'], 'all');
											}else{
												
												$order->add_order_note( sprintf(esc_html__( 'PDF didnt created for: %s. Reason: %s', 'iben-woo-elta-courier' ), $item, $pdf_created['result'] ) );
												$errorpdf .= '';
												$errorpdf.= esc_html__('This PDF didnt created.', 'iben-woo-elta-courier');
				
											}
										}
									}//foreach
				
				
									$output.= $this->iben_display_html_vouchers('time', json_encode($get_vouchers['elta_delivery_vouchers']), $order_id );
				
									if ($count_vouchers > 1 && $errorpdf == '' ){

										$random_string = $this->iben_generateRandomString();
										$name_pdf = $order_id.$random_string;

										$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
										$pdf->merge('file', $merge, 'P');
				
										$output .= iben_elta_Compined_Voucher_Link($name_pdf);
				
										global $wpdb;
										$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
							
										$wpdb->update( 
											$table_name, 
											array( 
												'elta_delivery_compined_vouchers' => $name_pdf
											), 
											array( 'id_order' => $order_id, 'id_elta_reason' => 1 )
										);

										$default2 = array(											
											'logged_user' => 'automatic',
											'id_user' => -1,
											'id_arrays' => json_encode($order_id),
											'voucher_arrays' => json_encode($get_vouchers['elta_delivery_vouchers']),
											'compined_pdf' => $name_pdf,
											'time' => current_time('timestamp')
										);
										$table_name2 = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
										$result = $wpdb->insert($table_name2, $default2);

									}
			
				
								}//if $get_vouchers exists
			
								$order->add_order_note( sprintf(esc_html__( 'Voucher: %s', 'iben-woo-elta-courier' ), $output ) );
								
							}else{
			
								$order->add_order_note( sprintf(esc_html__( 'Voucher created with no PDF: %s', 'iben-woo-elta-courier' ), $note ) );
			
							}


							// if setting. Set order completed, if enabled
							if( isset( $main_settings['iben_elta_main_settings_voucher_change_order_status'] ) && $main_settings['iben_elta_main_settings_voucher_change_order_status'] != -1) {

								$status_value = iben_elta_get_order_status_value($main_settings['iben_elta_main_settings_voucher_change_order_status'] );		
								$order->update_status( $status_value );						
		
							}

							$order->save();

							// if setting. Send admin email
							if( isset( $main_settings['iben_elta_main_settings_auto_createv_resend_admin_email'] ) ) {
								
								add_filter( 'woocommerce_new_order_email_allows_resend', '__return_true' );

								$mailer = WC()->mailer()->get_emails();
			
								$subject = apply_filters( 'iben_elta_auto_createv_subject_resend_admin_email_filter', sprintf(esc_html__( 'ELTA vouchers Completed #%s', 'iben-woo-elta-courier' ), $order->get_order_number() , $order_id) );
								$mailer['WC_Email_New_Order']->subject = $subject;
								$mailer['WC_Email_New_Order']->settings['subject'] = $subject;
								$mailer['WC_Email_New_Order']->settings['heading'] = $subject;
								$mailer['WC_Email_New_Order']->trigger( $order_id );
							}

							

							// if setting. Send completed email
							if( isset( $main_settings['iben_elta_main_settings_auto_createv_send_completed_email'] ) ) {
			
								$mailer = WC()->mailer()->get_emails();
								$mailer['WC_Email_Customer_Completed_Order']->trigger( $order_id );
								
							}
							
							// DOKAN compaitibility
							// Send vendor admin email
							$send_dokan_vendor_email = apply_filters('iben_elta_filter_send_dokan_vendor_admin_email', true);
							if ( class_exists('WeDevs_Dokan') && $send_dokan_vendor_email ){

								$mailer = WC()->mailer()->get_emails();
								$subject = apply_filters( 'iben_elta_auto_createv_subject_resend_vendor_admin_email_filter', sprintf(esc_html__( 'Vendor Vouchers: %s', 'iben-woo-elta-courier' ), $order->get_order_number() , $order_id) );
								$mailer['Dokan_Email_New_Order']->subject = $subject;
								$mailer['Dokan_Email_New_Order']->settings['subject'] = $subject;
								$mailer['Dokan_Email_New_Order']->settings['heading'] = $subject;
								$mailer['Dokan_Email_New_Order']->trigger( $order_id );
							}
							
						} else if ($webservice_run['state'] === false){
					
							$webservice_run = $webservice_run['result'];
			
							$order->add_order_note( sprintf(esc_html__( 'Error creating voucher: %s', 'iben-woo-elta-courier' ), $webservice_run ) );
						}
						$order->save();


					} // foreach -> $order_id

					if( isset( $main_settings['iben_elta_main_settings_auto_createv_resend_admin_email'] ) ) {
						add_filter( 'woocommerce_new_order_email_allows_resend', '__return_true' );
						if (!empty($parent_order_ids)){
							foreach($parent_order_ids as $order_id ) {

								$mailer = WC()->mailer()->get_emails();
								$subject = apply_filters( 'iben_elta_auto_createv_subject_resend_admin_email_filter', sprintf(esc_html__( 'ELTA vouchers Completed #%s', 'iben-woo-elta-courier' ), $order->get_order_number() , $order_id) );
								$mailer['WC_Email_New_Order']->subject = $subject;
								$mailer['WC_Email_New_Order']->settings['subject'] = $subject;
								$mailer['WC_Email_New_Order']->settings['heading'] = $subject;
								$mailer['WC_Email_New_Order']->trigger( $order_id );
			
							}
						} //!empty($parent_order_ids)
					}



				}//foreach -> $record
			
			
			
			
			}// if queries


		}

		/**
		 * 
		 *
		 * @since     1.10.0
		 */
		function iben_elta_create_voucher_by_orderid($id_order, $args = array()){

			$main_settings = $this->main_settings;

			if ( $id_order ) {
				$order = wc_get_order( absint($id_order) );
			}else{
				return;
			}
			if ( !is_a( $order, 'WC_Order' )  ){
				return;
			}

			$order_id = $order->get_id();

			if (1==1){

				$first_name = '';
				if( $order->get_billing_first_name() != $order->get_shipping_first_name() && !empty($order->get_shipping_first_name()) ) {
					$first_name = $order->get_shipping_first_name();
				} else {
					$first_name = $order->get_billing_first_name();
				}
				
				$last_name = '';
				if( $order->get_billing_last_name() != $order->get_shipping_last_name() && !empty($order->get_shipping_last_name()) ) {
					$last_name = $order->get_shipping_last_name();
				} else {
					$last_name = $order->get_billing_last_name();
				}
				
				$company = '';

				if( $order->get_billing_company() != $order->get_shipping_company() && !empty($order->get_shipping_company()) ) {
					$company = $order->get_shipping_company();
				} else {
					$company = $order->get_billing_company();
				}
				$company = ( $company == '' )? '':' ('.$company.')';
				

				$fullname = $last_name.' '.$first_name.$company; 
				$fullname = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($fullname, 'UTF-8') : $fullname;
				$fullname = apply_filters( 'iben_elta_filter_automatic_fullname_metafield_order', $fullname, $order_id );

				$address_1 = '';
				$address_2 = '';
				if( $order->get_billing_address_1() != $order->get_shipping_address_1() && !empty($order->get_shipping_address_1()) ) {
					$address_1 = $order->get_shipping_address_1();
				} else {
					$address_1 = $order->get_billing_address_1();
				}


				if( $order->get_billing_address_2() != $order->get_shipping_address_2() && !empty($order->get_shipping_address_2()) ) {
					$address_2 = $order->get_shipping_address_2();
				} else {
					$address_2 = $order->get_billing_address_2();
				}

			
				$display_address1 = apply_filters( 'iben_elta_filter_automatic_display_address1', true);
				$display_address2 = apply_filters( 'iben_elta_filter_automatic_display_address2', true);
				

				$fulladdress = '';

				if ($display_address1){
					$fulladdress.= $address_1;
				}

				if ( isset($address_2)  && $address_2 != '' ){

					if (!$display_address1 && $display_address2){
						$fulladdress.= $address_2;
					}else{
						$fulladdress.= ', '.$address_2;
					}

				}
				$fulladdress = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($fulladdress, 'UTF-8') : $fulladdress;
				$fulladdress = apply_filters( 'iben_elta_filter_automatic_fulladdress_metafield_order', $fulladdress, $order_id );

				$billing_country = $order->get_billing_country();
				$shipping_country = $order->get_shipping_country();


				if ( $billing_country != $shipping_country && $shipping_country != '' ){
					$country = $shipping_country;
				}else{
					$country = $billing_country;
				}


				$billing_state = $order->get_billing_state();
				$shipping_state = $order->get_shipping_state();

				if ( $billing_state != $shipping_state  && $shipping_state != '' ){
					$state = $shipping_state;
				}else{
					$state = $billing_state;
				}

				if ($country){
					if ($state){
						$state =  WC()->countries->get_states( $country )[$state];
					}else{
						$state = '';
					}
				}else{

					if ($state){
						$state =  WC()->countries->get_states( 'GR' )[$state];
					}else{
						$state = '';
					}

				}

				if ($order->get_billing_city() != $order->get_shipping_city()){
					$city = $order->get_shipping_city();
				}else{
					$city = $order->get_billing_city();
				}

				
				$display_city = apply_filters( 'iben_elta_filter_automatic_display_city', true);
				$display_state = apply_filters( 'iben_elta_filter_automatic_display_state', true);

				$region = '';

				if ($display_city){
					$region.= $city;
				}

				if (!$display_city && $display_state){
					$region.= $state;
				}else{
					$region.= ', '.$state;
				}
				$region = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($region, 'UTF-8') : $region;
				$region = apply_filters( 'iben_elta_filter_automatic_region_metafield_order', $region, $order_id );

				$postcode = '';
				if( $order->get_billing_postcode() != $order->get_shipping_postcode() && !empty($order->get_shipping_postcode()) ) {
					$postcode = $order->get_shipping_postcode();
				} else {
					$postcode = $order->get_billing_postcode();
				}
				$postcode = str_replace(array(' ', '-'), '', $postcode ?? '');
				$postcode = apply_filters( 'iben_elta_filter_automatic_postcode_metafield_order', $postcode, $order_id );


				$phone1 = '';

				$billing_phone1 = apply_filters('iben_elta_phone_format', $order->get_billing_phone(), $order_id);
				$billing_phone1 = str_replace(array(' ', '-', '.', '+'), '', $billing_phone1 ?? '');

				$shipping_phone1 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_automatic_phone1_metafield_order', 'shipping_phone' ), true));
				$shipping_phone1 = apply_filters('iben_elta_phone_format', $shipping_phone1, $order_id);
				$shipping_phone1 = str_replace(array(' ', '-', '.', '+'), '', $shipping_phone1 ?? '');

				if ( $billing_phone1 != $shipping_phone1 && $shipping_phone1 != ''){
					$phone1 = $shipping_phone1;
				}else{
					$phone1 = $billing_phone1;
				}
				
				$phone1 = apply_filters( 'iben_elta_filter_automatic_phone1', $phone1, $order_id);

				$phone2 = '';

				$billing_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_automatic_phone2_metafield_order', 'billing_mobile' ), true));
				$billing_phone2 = apply_filters('iben_elta_phone_format', $billing_phone2, $order_id);

				$shipping_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_automatic_shipping_phone2_metafield_order', 'shipping_mobile' ), true));
				$shipping_phone2 = apply_filters('iben_elta_phone_format', $shipping_phone2, $order_id);

				if ( $billing_phone2 != $shipping_phone2 && $shipping_phone2 != '' ){
					$phone2 = $shipping_phone2;
				}else{
					$phone2 = $billing_phone2;
				}

				$phone2 = apply_filters( 'iben_elta_filter_automatic_phone2', $phone2, $order_id);
	

				$comments = $order->get_customer_note();
				$comments = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($comments, 'UTF-8') : $comments;
				$comments = apply_filters( 'iben_elta_filter_automatic_comments_metafield_order', $comments, $order_id );
				

				$payment = $order->get_payment_method();
				
				//compatible with WC cod and 3rd party COD with name as id_codfee
				$payments = array('cod', 'id_codfee');
				
				$payments_filter = array();
				$payments_filter = apply_filters( 'iben_elta_filter_cashdelivery_payments_order_metabox', $payments_filter );
				
				if(is_array($payments_filter) && !empty($payments_filter)){
					foreach ($payments_filter as $payment_filter){
						$payments[] = $payment_filter;
					}
				}

				if (in_array($payment, $payments, true)) {
					$cashdelivery = $order->get_total();	
				}else{
					$cashdelivery = '0'; 
				}			
				$cashdelivery = apply_filters( 'iben_elta_filter_automatic_cash_metafield_order', $cashdelivery, $order_id );
				
				
				$weight = 0;
				// check if default weight has been set and use this
				$default_weight = sanitize_text_field($main_settings['iben_elta_main_settings_default_weight']);

				if( isset( $default_weight ) && $default_weight > 0 ) {
					
					$weight = $default_weight;

				}else{
			
					$total_weight = 0;
					foreach( $order->get_items() as $item_id => $product_item ){
						$quantity = $product_item->get_quantity(); // get quantity
						$product = $product_item->get_product(); // get the WC_Product object
						if( $product ) {
							$product_weight = $product->get_weight(); // get the product weight
							$total_weight += floatval( ( $product_weight ? $product_weight : 0 ) * $quantity );
							}
						
					}
					$weight = round($total_weight, 3);

					$unit = get_option('woocommerce_weight_unit');

					if ($unit == 'g'){
						$weight = $weight/1000;
					}
				}
				
				$weight = apply_filters( 'iben_elta_filter_automatic_weight_metafield_order', $weight, $order_id );


				$parcels = '1'; 

				$ref = '';
				if (isset( $main_settings['iben_elta_main_settings_enable_ref_order'] ) ){
					$ref = $order->get_order_number();
				}			
				$ref = apply_filters( 'iben_elta_filter_automatic_ref_metafield_order', $ref, $order_id );
			}

			$types['fullname'] = sanitize_text_field( $fullname ); 
			$types['address'] = sanitize_text_field( $fulladdress );
			$types['region'] = sanitize_text_field( $region );						
			$types['postal_code'] = sanitize_text_field( $postcode );
			$types['phone1'] = sanitize_text_field( $phone1 );
			$types['phone2'] = sanitize_text_field( $phone2 );
			$types['service'] = sanitize_text_field(1);
			$types['weight'] = sanitize_text_field( $weight );
			$types['packages_number'] = sanitize_text_field( $parcels );
			$types['subcode'] = sanitize_text_field( '' ); 

			$types['comments'] = sanitize_text_field( $comments );
			$types['cash_delivery'] = sanitize_text_field( $cashdelivery );
			$types['sur_1'] = sanitize_text_field(0);
			$types['sur_2'] = sanitize_text_field(0);
			$types['sur_3'] = sanitize_text_field(0);
			$types['ref'] = sanitize_text_field($ref);
			$types['second_account'] = sanitize_text_field( 1 );
			
			$types = apply_filters( 'iben_elta_filter_multi_create_types_order_metabox', $types, $order_id );

			$webservice_run = $this->ibenEltaWeServiceCreateVoucher($types);


			if ($webservice_run['state'] === false){
		
				$webservice_run = $webservice_run['result'];

				$order->add_order_note( sprintf(esc_html__( 'Error creating voucher: %s', 'iben-woo-elta-courier' ), $webservice_run ) );
				return $webservice_run;
			}


			// success! Voucher has been created here
			if ($webservice_run['state'] === true){
				
				$export = array();

				$webservice_run = $webservice_run['result'];
			
				$getVg_child = array();
				array_push($getVg_child, $webservice_run->getVg_code());
				$main_voucher = $webservice_run->getVg_code();

				if ( !empty($webservice_run->getVg_child()) ){
					$child_vouchers = array_filter($webservice_run->getVg_child());

					if( !empty($child_vouchers) && sizeof($child_vouchers) != 0) {

						foreach ($child_vouchers as $voucher){
							array_push($getVg_child, $voucher);
						}
					
					}
				}

				$default = array(
					'logged_user' => 'Automatic',
					'id_user' => '-1',
					'id_order' => $order_id,
					'id_order_parent' => -1,
					'id_elta_reason' => 1,
					'elta_returned_vouchers' => '',
					'elta_delivery_vouchers' => json_encode($getVg_child),
					'elta_pdf_type' => $this->iben_elta_main_settings_papersize,
					'elta_voucher_info' => maybe_serialize($types),
					'id_elta_account' => $types['second_account'],
					'duplicated_record' => 0,
					'time' => current_time('timestamp'),
					'plugin_version' => $this->version
				);

				global $wpdb;
				$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
				$result = $wpdb->insert($table_name, $default);


				do_action( 'iben_elta_hook_voucher_created', $order_id, $main_voucher );

				$order->update_meta_data( '_iben_elta_voucher_data', $types );
				$order->update_meta_data( '_iben_elta_voucher_ship_number', $main_voucher );
				
			
				$export['status'] = 1;
				$export['main_voucher'] = $main_voucher;
				$export['query'] = $default;

	
				if ( !empty($getVg_child) && $getVg_child != ''){

					$order->update_meta_data( '_iben_elta_voucher_generated_vouchers', json_encode($getVg_child) );

				}

				
				// always before SEND the email. 
				// if setting. Attach tracking info is enabled
				if( isset( $args['attach_tracking'] ) && $args['attach_tracking'] === true ) {

					$order->update_meta_data( 'iben_elta_tracking_order_notification', 1 );

				}else{

					$order->update_meta_data( 'iben_elta_tracking_order_notification', 0 );
				}

				
				$errorpdf = '';
				//if setting. Create PDF is enabled
				$create_pdf_flag = apply_filters('iben_elta_filter_trigger_create_create_pdf_flag', false);

				if( $create_pdf_flag || isset( $args['create_pdf_flag'] ) && $args['create_pdf_flag'] === true ) {

					$get_vouchers = iben_elta_get_Array_Vouchers_from_id_order($order_id);

					if( !empty($get_vouchers) || sizeof( $get_vouchers ) != 0 ){
		

						$count_vouchers = count($get_vouchers['elta_delivery_vouchers']) ;
	
						require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
						$pdf = new \Clegginabox\PDFMerger\PDFMerger;
						$output = '';
	
						foreach ($get_vouchers['elta_delivery_vouchers'] as $item) {

							$file2 = IBEN_WOO_ELTA_UPLOAD_FOLDER.$item.".pdf";                            
							
							if ($item != '' ){

								$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($item), $types['second_account']);
	
								if ( $pdf_created['state'] === true){
	
									$pdf->addPDF($pdf_created['result'], 'all');

								}else{
									
									$order->add_order_note( sprintf(esc_html__( 'PDF didnt created for: %s. Reason: %s', 'iben-woo-elta-courier' ), $item, $pdf_created['result'] ) );
									$errorpdf .= '';
									$errorpdf.= esc_html__('This PDF didnt created.', 'iben-woo-elta-courier');
	
								}
							}
						}//foreach
	
	
						$output.= $this->iben_display_html_vouchers('time', json_encode($get_vouchers['elta_delivery_vouchers']), $order_id );
	
						global $wpdb;
						$table_name = $wpdb->prefix . 'iben_elta_vouchers';	

						if ($count_vouchers > 1 && $errorpdf == '' ){

							$random_string = $this->iben_generateRandomString();
							$name_pdf = $order_id.$random_string;

							$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
							$pdf->merge('file', $merge, 'P');
	
							$output .= iben_elta_Compined_Voucher_Link($name_pdf);
					
							$wpdb->update( 
								$table_name, 
								array( 
									'elta_delivery_compined_vouchers' => $name_pdf
								), 
								array( 'id_order' => $order_id, 'id_elta_reason' => 1 )
							);

							$default2 = array(											
								'logged_user' => 'automatic',
								'id_user' => -1,
								'id_arrays' => json_encode($order_id),
								'voucher_arrays' => json_encode($get_vouchers['elta_delivery_vouchers']),
								'compined_pdf' => $name_pdf,
								'time' => current_time('timestamp')
							);
							$table_name2 = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
							$result = $wpdb->insert($table_name2, $default2);

						}

	
					}//if $get_vouchers exists

					$order->add_order_note( sprintf(esc_html__( 'Voucher: %s', 'iben-woo-elta-courier' ), $output ) );
					
				}


				// if setting. Set order completed, if enabled
				if( isset( $args['change_order_status'] ) && $args['change_order_status'] != '' ) {

					$status_value = iben_elta_get_order_status_value( $args['change_order_status'] );		
					$order->update_status( $status_value );						

				}	
				
				$order->save();

				$order->save();

				// if setting. Send admin email
				if( isset( $args['resend_admin_email'] ) && $args['resend_admin_email'] === true ) {
					$mailer = WC()->mailer()->get_emails();

					$subject = apply_filters( 'iben_elta_auto_createv_subject_resend_admin_email_filter', sprintf(esc_html__( 'ELTA vouchers Completed #%s', 'iben-woo-elta-courier' ), $order->get_order_number() , $order_id) );
					$mailer['WC_Email_New_Order']->subject = $subject;
					$mailer['WC_Email_New_Order']->settings['subject'] = $subject;
					$mailer['WC_Email_New_Order']->settings['heading'] = $subject;
					$mailer['WC_Email_New_Order']->trigger( $order_id );
				}

				// if setting. Send completed email
				if( isset( $args['resend_completed_email'] ) && $args['resend_completed_email'] === true) {

					$mailer = WC()->mailer()->get_emails();
					$mailer['WC_Email_Customer_Completed_Order']->trigger( $order_id );
					
				}

				return $export;

				
			}			

		}


		/**
		 * Disable default status change processing -> completed email
		 *
		 * @since     1.6.0
		 */
		function iben_elta_disable_order_status_completed_notification( $email_class ) {

			// Completed order emails
			remove_action( 'woocommerce_order_status_completed_notification', array( $email_class->emails['WC_Email_Customer_Completed_Order'], 'trigger' ) );
		
		}


		/** 
		*
		* @since     1.7.5
		*/
		public function iben_elta_bulk_create_vouchers_ajax_handler(){
			
			if (isset($_POST['iben_data'])) {
				
				// Get array data. Values being secured later
				$data = $_POST['iben_data'];			

				// NONCE IN AJAX URL
				$nonce_name   = isset( $data['iben_elta_nonce'] ) ? sanitize_text_field( $data['iben_elta_nonce'] ) : '';
				$nonce_action = 'iben_elta_nonce';
			
				// // Check if nonce is valid.
				if ( ! wp_verify_nonce( $nonce_name, $nonce_action ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'WP Security token failure', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}  

				if ( ! current_user_can( 'edit_shop_orders' ) ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You cannot edit shop orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$output = '';

				// it secured by absint later
				$id_orders = $data['id_arrays'];

				if ( !isset($id_orders) || count($id_orders) == 0 ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'You have to check some orders', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}				
			

				// DOKAN compatibility here
				$sub_order_ids  = array();
				$parent_order_ids = array();
				

				foreach ($id_orders as $before_order_id) {

					$order = wc_get_order( absint($before_order_id) );
					if (!is_a( $order, 'WC_Order' )){
						continue;
					}

					$get_vouchers = iben_elta_get_Vouchers_from_id_order($before_order_id);
					if ( !empty($get_vouchers) ){
						continue;
					}
					
					$has_suborder = $order->get_meta( 'has_sub_order' );

					// DOKAN compatibility here
					if ( $has_suborder ) {
						
						$parent_order_ids[] = $before_order_id;

						$sub_order_ids = get_children(
							[
								'post_parent' => $order->get_id(),
								'post_type' => 'shop_order',
								'fields' => 'ids',
							]
						);

					} else {
						$sub_order_ids[] = $before_order_id;
					}

				}//each id order


				if ( !isset($sub_order_ids) || count($sub_order_ids) == 0 ) {
					$resultss = array (
						"status" => "failure",
						"result" => esc_html__( 'There are no orders to create a voucher', 'iben-woo-elta-courier' )
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$output = '';
				$id_orders_with_vouchers = array();
				$vouchers_created_number = 0;
				$log = '';

				foreach($sub_order_ids as $order_id ) {

					$order = wc_get_order( $order_id );

					if (!is_a( $order, 'WC_Order' )){
						continue;
					}					
							
					$user_subcode = '';
					//DOKAN compatibility here
					if (function_exists('dokan_get_seller_id_by_order')) {
						$seller_id = dokan_get_seller_id_by_order( $order_id );	
						$user_subcode = get_user_meta($seller_id, 'iben_elta_user_subcode_meta', true);			
						if ($user_subcode != '' ){
							$user_subcode = $user_subcode;
						}
					}
					
					$parent_id = $order->get_parent_id();
		
					if ( $parent_id ) {
						$parent_id = $parent_id;
					}else{
						$parent_id = 0;
					}						

					if (1==1){

						$first_name = '';
						if( $order->get_billing_first_name() != $order->get_shipping_first_name() && !empty($order->get_shipping_first_name()) ) {
							$first_name = $order->get_shipping_first_name();
						} else {
							$first_name = $order->get_billing_first_name();
						}
						
						$last_name = '';
						if( $order->get_billing_last_name() != $order->get_shipping_last_name() && !empty($order->get_shipping_last_name()) ) {
							$last_name = $order->get_shipping_last_name();
						} else {
							$last_name = $order->get_billing_last_name();
						}

						$company = '';
						if( $order->get_billing_company() != $order->get_shipping_company() && !empty($order->get_shipping_company()) ) {
							$company = $order->get_shipping_company();
						} else {
							$company = $order->get_billing_company();
						}
						$company = ( $company == '' )? '':' ('.$company.')';
						

						$fullname = $last_name.' '.$first_name.$company;
						$fullname = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($fullname, 'UTF-8') : $fullname;
						
						$fullname = apply_filters( 'iben_elta_filter_multi_create_fullname_order_metabox', $fullname, $order_id );

						// Address
						$address_1 = '';
						$address_2 = '';
						if( $order->get_billing_address_1() != $order->get_shipping_address_1() && !empty($order->get_shipping_address_1()) ) {
							$address_1 = $order->get_shipping_address_1();
						} else {
							$address_1 = $order->get_billing_address_1();
						}


						if( $order->get_billing_address_2() != $order->get_shipping_address_2() && !empty($order->get_shipping_address_2()) ) {
							$address_2 = $order->get_shipping_address_2();
						} else {
							$address_2 = $order->get_billing_address_2();
						}


						$display_address1 = apply_filters( 'iben_elta_filter_multi_create_display_address1', true);
						$display_address2 = apply_filters( 'iben_elta_filter_multi_create_display_address2', true);
						
		
						$fulladdress = '';
		
						if ($display_address1){
							$fulladdress.= $address_1;
						}
		
						if ( isset($address_2)  && $address_2 != '' ){
		
							if (!$display_address1 && $display_address2){
								$fulladdress.= $address_2;
							}else{
								$fulladdress.= ', '.$address_2;
							}		
						}
						$fulladdress = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($fulladdress, 'UTF-8') : $fulladdress;
						$fulladdress = apply_filters( 'iben_elta_filter_multi_create_fulladdress_order_metabox', $fulladdress, $order_id );

						$billing_country = $order->get_billing_country();
						$shipping_country = $order->get_shipping_country();
		
		
						if ( $billing_country != $shipping_country && $shipping_country != '' ){
							$country = $shipping_country;
						}else{
							$country = $billing_country;
						}
		
		
						$billing_state = $order->get_billing_state();
						$shipping_state = $order->get_shipping_state();
		
						if ( $billing_state != $shipping_state  && $shipping_state != '' ){
							$state = $shipping_state;
						}else{
							$state = $billing_state;
						}
		
						if ($country){
							if ($state){
								$state =  WC()->countries->get_states( $country )[$state];
							}else{
								$state = '';
							}
						}else{
		
							if ($state){
								$state =  WC()->countries->get_states( 'GR' )[$state];
							}else{
								$state = '';
							}
		
						}
						if ($order->get_billing_city() != $order->get_shipping_city()){
							$city = $order->get_shipping_city();
						}else{
							$city = $order->get_billing_city();
						}
						
						$display_city = apply_filters( 'iben_elta_filter_multi_create_display_city', true);
						$display_state = apply_filters( 'iben_elta_filter_multi_create_display_state', true);
		
						$region = '';
		
						if ($display_city){
							$region.= $city;
						}
		
						if (!$display_city && $display_state){
							$region.= $state;
						}else{
							$region.= ', '.$state;
						}
						$region = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($region, 'UTF-8') : $region;
						$region = apply_filters( 'iben_elta_filter_multi_create_region_order_metabox', $region, $order_id );

						// Postcode
						$postcode = '';
						if( $order->get_billing_postcode() != $order->get_shipping_postcode() && !empty($order->get_shipping_postcode()) ) {
							$postcode = $order->get_shipping_postcode();
						} else {
							$postcode = $order->get_billing_postcode();
						}
						$postcode = str_replace(array(' ', '-'), '', $postcode ?? '');
						$postcode = apply_filters( 'iben_elta_filter_multi_create_postcode_order_metabox', $postcode, $order_id );
		

						$phone1 = '';
		
						$billing_phone1 = apply_filters('iben_elta_phone_format', $order->get_billing_phone(), $order_id);
						$billing_phone1 = str_replace(array(' ', '-', '.', '+'), '', $billing_phone1 ?? '');

						$shipping_phone1 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_multi_create_phone1_metafield_order', 'shipping_phone' ), true));
						$shipping_phone1 = apply_filters('iben_elta_phone_format', $shipping_phone1, $order_id);
						$shipping_phone1 = str_replace(array(' ', '-', '.', '+'), '', $shipping_phone1 ?? '');

						if ( $billing_phone1 != $shipping_phone1 && $shipping_phone1 != ''){
							$phone1 = $shipping_phone1;
						}else{
							$phone1 = $billing_phone1;
						}

						$phone1 = apply_filters( 'iben_elta_filter_automatic_phone1', $phone1, $order_id);

						$phone2 = '';
						$billing_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_multi_create_phone2_metafield_order', 'billing_mobile' ), true));
						$billing_phone2 = apply_filters('iben_elta_phone_format', $billing_phone2, $order_id);

						$shipping_phone2 = sanitize_text_field($order->get_meta( apply_filters( 'iben_elta_filter_multi_create_shipping_phone2_metafield_order', 'shipping_mobile' ), true));
						$shipping_phone2 = apply_filters('iben_elta_phone_format', $shipping_phone2, $order_id);
		
						if ( $billing_phone2 != $shipping_phone2 && $shipping_phone2 != '' ){
							$phone2 = $shipping_phone2;
						}else{
							$phone2 = $billing_phone2;
						}						
						$phone2 = apply_filters( 'iben_elta_filter_automatic_phone2', $phone2, $order_id);


						$comments = '';
						$enablecomments = apply_filters('iben_elta_filter_multi_create_enable_comments_metafield_order', false);
						if ($enablecomments){

							$comments = $order->get_customer_note();
							$comments = isset( $this->main_settings['iben_elta_main_settings_convert_uppercase'] ) ? iben_elta_upper_no_accents($comments, 'UTF-8') : $comments;
							$comments = apply_filters( 'iben_elta_filter_multi_create_comments_metafield_order', $comments, $order_id );
						}

						
						$payment = $order->get_payment_method();
						
						//compatible with WC cod and 3rd party COD with name as id_codfee
						$payments = array('cod', 'id_codfee');
						
						$payments_filter = array();
						$payments_filter = apply_filters( 'iben_elta_filter_cashdelivery_payments_order_metabox', $payments_filter );
						
						if(is_array($payments_filter) && !empty($payments_filter)){
							foreach ($payments_filter as $payment_filter){
								$payments[] = $payment_filter;
							}
						}	

						if (in_array($payment, $payments, true)) {
							$cashdelivery = $order->get_total();	
						}else{
							$cashdelivery = '0'; 
						}			
						$cashdelivery = apply_filters( 'iben_elta_filter_multi_create_cash_metafield_order', $cashdelivery, $order_id );														


						$weight = 0;
						// check if default weight has been set and use this
						$default_weight = sanitize_text_field($this->main_settings['iben_elta_main_settings_default_weight']);
		
						if( isset( $default_weight ) && $default_weight > 0 ) {
							
							$weight = $default_weight;
		
						}else{
					
							$total_weight = 0;
							foreach( $order->get_items() as $item_id => $product_item ){
								$quantity = $product_item->get_quantity(); // get quantity
								$product = $product_item->get_product(); // get the WC_Product object
								if( $product ) {
									$product_weight = $product->get_weight(); // get the product weight
									$total_weight += floatval( ( $product_weight ? $product_weight : 0 ) * $quantity );
									}
								
							}
							$weight = round($total_weight, 3);

							$unit = get_option('woocommerce_weight_unit');

							if ($unit == 'g'){
								$weight = $weight/1000;
							}

						}
						
						$weight = apply_filters( 'iben_elta_filter_multi_create_weight_metafield_order', $weight, $order_id );


						// Ref
						$ref = '';
						if ( isset($this->main_settings['iben_elta_main_settings_enable_ref_order']) ){
							$ref = $order->get_order_number();
							$ref = apply_filters( 'iben_elta_filter_multi_create_ref_metafield_order', $ref, $order_id );
						}
						
						$parcels = '1';

						$types['fullname'] = $fullname;
						$types['address'] = $fulladdress;
						$types['region'] = $region;
						$types['postal_code'] = $postcode;
						$types['phone1'] = $phone1;
						$types['phone2'] = $phone2;
						$types['service'] = '1';
						$types['weight'] = sanitize_text_field($weight);
						$types['comments'] = $comments;
						$types['packages_number'] = $parcels;
						$types['sur_1'] = '0';
						$types['sur_2'] = '0';
						$types['sur_3'] = '0';
						$types['subcode'] = sanitize_text_field( $user_subcode );
						$types['cash_delivery'] = sanitize_text_field($cashdelivery);
						$types['ref'] = $ref;
						$types['second_account'] = 1;
				
					}

					$types = apply_filters( 'iben_elta_filter_multi_create_types_order_metabox', $types, $order_id );
					
					$webservice_run = $this->ibenEltaWeServiceCreateVoucher($types);

					if ($webservice_run['state'] === false){
						$webservice_run = $webservice_run['result'];
						$order->add_order_note( sprintf(esc_html__( 'Error creating voucher through combined action: %s', 'iben-woo-elta-courier' ), $webservice_run ) );
						$log .= '<br/>#'.$order_id.' error <br/>';
						
						continue;
					}

					// success! Voucher has been created here
					if ($webservice_run['state'] === true){
						
						$webservice_run = $webservice_run['result'];
						
						$getVg_child = array();
						$main_voucher = $webservice_run->getVg_code();
						array_push($getVg_child, $webservice_run->getVg_code());
						array_push($id_orders_with_vouchers, $order_id);
						$vouchers_created_number++;

						if ( !empty($webservice_run->getVg_child()) ){
							$child_vouchers = array_filter($webservice_run->getVg_child());
		
							if( !empty($child_vouchers) && sizeof($child_vouchers) != 0) {
								foreach ($child_vouchers as $voucher){
									array_push($getVg_child, $voucher);
								}
							}
						}
						$current_user = wp_get_current_user();

						$default = array(							
							'logged_user' => esc_html( $current_user->user_login),
							'id_user' => get_current_user_id(),
							'id_order' => $order_id,
							'id_order_parent' => $parent_id,	
							'id_elta_reason' => 1,
							'elta_returned_vouchers' => '',
							'elta_delivery_vouchers' => json_encode($getVg_child),
							'elta_pdf_type' => $this->iben_elta_main_settings_papersize,
							'id_elta_account' => $types['second_account'],
							'elta_voucher_info' => maybe_serialize($types),
							'duplicated_record' => 0,
							'time' => current_time('timestamp'),
							'plugin_version' => $this->version,
						);
	
						global $wpdb;
						$table_name = $wpdb->prefix . 'iben_elta_vouchers';	
						$result = $wpdb->insert($table_name, $default);
				
						$order->add_order_note( esc_html__( 'ELTA Voucher created through multi action', 'iben-woo-elta-courier' ) );
						
						do_action( 'iben_elta_hook_voucher_created', $order_id, $main_voucher );

						$order->update_meta_data( '_iben_elta_voucher_data', $types );
						$order->update_meta_data( '_iben_elta_voucher_ship_number', $main_voucher );

						$get_vouchers = iben_elta_get_Vouchers_from_id_order($order_id);
						
						if ( !empty($getVg_child) && $getVg_child != ''){

							$order->update_meta_data( '_iben_elta_voucher_generated_vouchers', json_encode($getVg_child) );
						}

						// always before SEND the email. 
						// if setting. Attach tracking info is enabled
						if ( !$order->meta_exists('iben_elta_tracking_order_notification') ){
							
							$is_enabled = isset( $this->main_settings['iben_elta_main_settings_email_completed'] ) ? (int)$this->main_settings['iben_elta_main_settings_email_completed'] : 0;	
							$is_default = isset( $this->main_settings['iben_elta_main_settings_email_checked_default_completed'] ) ? (int)$this->main_settings['iben_elta_main_settings_email_checked_default_completed'] : 0;	

							if($is_enabled && $is_default){
								$order->update_meta_data( 'iben_elta_tracking_order_notification', 1 );
							}
						}						


						// if setting. Set order completed, if enabled
						if( isset( $this->main_settings['iben_elta_main_settings_voucher_change_order_status'] ) && $this->main_settings['iben_elta_main_settings_voucher_change_order_status'] != -1) {
							
							$status_value = iben_elta_get_order_status_value($this->main_settings['iben_elta_main_settings_voucher_change_order_status'] );
							
							$order->update_status( $status_value );


	
						}	
						
						$order->save();

						// if setting. Send completed email
						if( isset( $this->main_settings['iben_elta_button_settings_create_send_completed_email'] ) ) {
							$mailer = WC()->mailer()->get_emails();
							$mailer['WC_Email_Customer_Completed_Order']->trigger( $order_id );
						}				
						
					}
				} // foreach -> $order_id

				$output .= $vouchers_created_number.' '.esc_html__( 'vouchers created.', 'iben-woo-elta-courier' );
				
				
				if ( !isset($id_orders_with_vouchers) || count($id_orders_with_vouchers) == 0 ) {
					$_result = esc_html__( '0 voucher created. Check inside selected order for more information', 'iben-woo-elta-courier' );

					$resultss = array (
						"status" => "failure",
						"result" => $_result.$log
					);
					echo json_encode ( $resultss );
					wp_die();
				}

				$enable_print_pdf = apply_filters( 'iben_elta_filter_multi_create_enable_print_pdf', true);

				if ($enable_print_pdf){
					// reset output here
					$output = '';
					require_once IBEN_WOO_ELTA_ROOT_DIR . 'vendor/autoload.php';
					$pdf = new \Clegginabox\PDFMerger\PDFMerger;

					$voucher_list = array();
					$id_list = array();
					$pdf_exist = false;
					$number_pdfs = 0;
					foreach ($id_orders_with_vouchers as $id_order) {
						array_push($id_list, $id_order);

						$vouchers = iben_elta_get_Vouchers_from_id_order($id_order);

						if(!empty($vouchers) && count($vouchers) > 0){
							foreach ( $vouchers as $voucher) {

								$vouchers_inside = json_decode($voucher->elta_delivery_vouchers);
								foreach ( $vouchers_inside as $voucher_final) {
									array_push($voucher_list, $voucher_final);

									$pdf_created = $this->ibenEltaWebServiceGenerateVoucherPDF(strval($voucher_final), iben_elta_get_account_from_delivery_voucher($voucher_final) );
								
									if ( $pdf_created['state'] !== true){
										$errorpdf.= '('.$voucher_final.') <br/>'.esc_html__('This PDF didnt created. You have to Regenerate it!', 'iben-woo-elta-courier').' <span class="iben_elta_pdf_error">'.$pdf_created['result'].'</span>';
									}else{
										$pdf->addPDF($pdf_created['result'], 'all');
										$pdf_exist = true;
										$number_pdfs++;
									}

								}
								
							}

							if ($pdf_exist && $number_pdfs > 1){
								$random_string = $this->iben_generateRandomString();
								$name_pdf = $id_order.$random_string;
			
								$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
								
								$pdf->merge('file', $merge, 'P');
					
								$wpdb->update( 
									$table_name, 
									array( 
										'elta_delivery_compined_vouchers' => $name_pdf
									), 
									array( 'id_order' => $id_order, 'id_elta_reason' => 1 )
								);	
			
			
								$current_user = wp_get_current_user();
								$default2 = array(									
									'logged_user' => esc_html( $current_user->user_login),
									'id_user' => get_current_user_id(),
									'id_arrays' => json_encode($id_order),
									'voucher_arrays' => json_encode($voucher_list),
									'compined_pdf' => $name_pdf,
									'time' => current_time('timestamp')
								);
								$table_name2 = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
								$result = $wpdb->insert($table_name2, $default2);
							}

						}
						//reset for next bulk 
						$number_pdfs = 0;
					}

					 // compine multiple order to 1
					if ($pdf_exist){
						$random_string = $this->iben_generateRandomString();
						$name_pdf = $order_id.$random_string;

						$merge =  IBEN_WOO_ELTA_UPLOAD_FOLDER.$name_pdf.'.pdf';
						$teet = empty($pdf);


						$pdf->merge('file', $merge, 'P');

						$output .= iben_elta_Compined_Voucher_Link($name_pdf);


						$current_user = wp_get_current_user();
						$default2 = array(							
							'logged_user' => esc_html( $current_user->user_login),
							'id_user' => get_current_user_id(),
							'id_arrays' => json_encode($id_list),
							'voucher_arrays' => json_encode($voucher_list),
							'compined_pdf' => $name_pdf,
							'time' => current_time('timestamp')
						);
						$table_name2 = $wpdb->prefix . 'iben_elta_bulk_vouchers';	
						$result = $wpdb->insert($table_name2, $default2);
					}
				}

				$result = $output;

				if (isset($errorpdf)){
					$result = $output.'<br/>'.$errorpdf;
				}
	
				$message = 'Generated';

				$resultss2 = array (
					"status" => "success",
					"message" => $message,
					"result" => $result.$log
				);
		
				echo json_encode ( $resultss2 );				

			}

			wp_die();
		}


		/**
		 *
		 * @since     1.11.0
		*/
		function iben_elta_tracing_shortcode_handler($atts) {

			$attr = shortcode_atts( array(
				'order_id' => null,
			), $atts );
	
			$export = '';

			if (!$attr['order_id']) return;


			$get_vouchers = iben_elta_get_Vouchers_from_id_order($attr['order_id']);
			if (!empty($get_vouchers) && is_array($get_vouchers) ){

				foreach ($get_vouchers as $index => $voucher){

					$vouch = json_decode($voucher->elta_delivery_vouchers);
					$vouch = $vouch['0'];

					$check_tracing = $this->ibenEltaWebServiceTracing( $vouch, '1', $voucher->id_elta_account);

					if ($check_tracing['state'] === true){
						
						$check_tracing = $check_tracing['result'];
						
						
						$checkpoints = $check_tracing->getWeb_status();

						if (is_array($checkpoints) && count($checkpoints) > 0){
							$export .= '<ul>';
							$export .= "<li><span class='iben_elta_front_tracing voucher'>".esc_html__( 'Tracking number: ', 'iben-woo-elta-courier' )."{$vouch}</span> </li>";
							foreach ( $checkpoints as $item){
								if( !$item->getWeb_status_title() ) {
									continue;
								}
								$date = substr($item->getWeb_date(), 6, 2).'-'.substr($item->getWeb_date(), 4, 2).'-'.substr($item->getWeb_date(), 0, 4);
								$time = substr($item->getWeb_time(), 0, 2).':'.substr($item->getWeb_time(), 2, 2);					

								$export .= "<li>
								<span class='iben_elta_front_tracing iben_elta_front_title Status'>{$item->getWeb_status_title()}</span>
								<span class='iben_elta_front_tracing iben_elta_front_datetime StatusDate'>".$date." ".$time."</span>

								</li>";
							}
							$export .= '</ul>';
						}						

					}
				}
			}

	
			return $export;
		}

		/**
		 *
		 * @since     1.11.0
		*/
		function iben_elta_tracking_number_shortcode_handler($atts) {

			$attr = shortcode_atts( array(
				'order_id' => null,
				'prefix_text' => null,
				'suffix_text' => null,
			), $atts );

			$allowed_html = array(
				'a' => array(
					'href' => array(),
					'title' => array(),
					'alt' => array(),
				),
				'br' => array(),
				'em' => array(),
				'strong' => array(),
				'ul' => array(),
				'li' => array(),
				'ol' => array(),
				'p' => array(),
				'h1' => array(),
				'h2' => array(),
				'h3' => array(),
				'h4' => array(),
				'h5' => array(),
				'h6' => array(),
				'i' => array(),
			);

			$export = '';

			if (!$attr['order_id']) return;

			$get_vouchers = iben_elta_get_Vouchers_from_id_order($attr['order_id']);
			if (!empty($get_vouchers) && is_array($get_vouchers) ){

				$export .= !empty($attr['prefix_text']) ? wp_kses($attr['prefix_text'], $allowed_html) : '';
				foreach ($get_vouchers as $record){
					$voucher = json_decode($record->elta_delivery_vouchers);
					$voucher = $voucher['0'];
					$export .= '<a href="https://www.elta-courier.gr/search?br='.$voucher.'" target="_blank">'.$voucher.'</a> ';
				}
				$export .= !empty($attr['suffix_text']) ? wp_kses($attr['suffix_text'], $allowed_html) : '';
			}

	
			return $export;
		}	
		
		/**
		 *
		 * @since     2.4.0
		*/
		function iben_elta_courier_name_shortcode_handler($atts) {

			$attr = shortcode_atts( array(
				'order_id' => null,
				'prefix_text' => null,
				'suffix_text' => null,
			), $atts );

			$allowed_html = array(
				'a' => array(
					'href' => array(),
					'title' => array(),
					'alt' => array(),
				),
				'br' => array(),
				'em' => array(),
				'strong' => array(),
				'ul' => array(),
				'li' => array(),
				'ol' => array(),
				'p' => array(),
				'h1' => array(),
				'h2' => array(),
				'h3' => array(),
				'h4' => array(),
				'h5' => array(),
				'h6' => array(),
				'i' => array(),
			);

			$export = '';

			if (!$attr['order_id']) return;

			$get_vouchers = iben_elta_get_Vouchers_from_id_order($attr['order_id']);
			if (!empty($get_vouchers) && is_array($get_vouchers) ){

				$export .= !empty($attr['prefix_text']) ? wp_kses($attr['prefix_text'], $allowed_html) : '';

				$export .= apply_filters('iben_elta_courier_name_shortcode_filter', esc_html__( 'ΕΛΤΑ Courier', 'iben-woo-elta-courier' ) );

				$export .= !empty($attr['suffix_text']) ? wp_kses($attr['suffix_text'], $allowed_html) : '';
			}
	
			return $export;
		}




	}
}